<?php
//-------------------------------------------------------------------------
// OVIDENTIA http://www.ovidentia.org
// Ovidentia is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.
//-------------------------------------------------------------------------
/**
 * @license http://opensource.org/licenses/gpl-license.php GNU General Public License (GPL)
 * @copyright Copyright (c) 2006 by CANTICO ({@link http://www.cantico.fr})
 */
include_once 'base.php';

define('FUNC_WIDGETS_PHP_PATH', $GLOBALS['babInstallPath'] . 'addons/widgets/widgets/');
define('FUNC_WIDGETS_JS_PATH', $GLOBALS['babInstallPath'] . 'skins/ovidentia/templates/addons/widgets/');


/**
 * Provides a user interface widgets factory.
 */
class Func_Widgets extends bab_Functionality
{
	const CLASS_PREFIX = 'Widget_';
	const CLASS_PREFIX_LENGTH = 7;


	/**
	 * @return string
	 *
	 */
	public function getDescription()
	{
		return 'Provides user interface widgets.';
	}

	/**
	 * Register myself as a functionality.
	 * @static
	 */
	public static function register()
	{
		require_once $GLOBALS['babInstallPath'].'utilit/functionalityincl.php';
		$functionalities = new bab_functionalities();
		$functionalities->registerClass(__CLASS__, __FILE__);
	}


	/**
	 * Includes all necessary CSS files to the current page.
	 *
	 * @return bool		false in case of error
	 */
	public function includeCss()
	{
		global $babBody;
		$babBody->addStyleSheet('addons/widgets/widgets.css');
		return true;
	}



	/**
	 * Includes the definition of the class $itemClass.
	 *
	 * @param string	$itemClass		The classname.
	 */
	public function includePhpClass($itemClass)
	{
		$itemClassPhpFilename = strtolower(substr($itemClass, Func_Widgets::CLASS_PREFIX_LENGTH)) . '.class.php';
		include_once FUNC_WIDGETS_PHP_PATH . $itemClassPhpFilename;
	}

	/**
	 * Creates and returns a new item of type $itemClass.
	 *
	 * @param string	$itemClass		The classname of the item to create.
	 * @param mixed		...				Additional parameters depending on item class.
	 * @return Widget_Item					The newly created item or null.
	 */
	public function create($itemClass)
	{
		Func_Widgets::includePhpClass($itemClass);
		if (!class_exists($itemClass)) {
			return null;
		}
		$args = func_get_args();
		array_shift($args);
		$widget = call_user_func_array($itemClass, $args);
		return $widget;
	}


	/**
	 * Constructs a Widget_Color.
	 *
	 * @return Widget_Color
	 */
	public function Color()
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'color.class.php';
		return new Widget_Color();
	}

	/**
	 * Constructs a Widget_Layout.
	 *
	 * @param string $id	The item unique id.
	 * @return Widget_Layout
	 */
	public function Layout($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'layout.class.php';
		return new Widget_Layout($id);
	}




	/**
	 * Constructs a Widget_ListLayout.
	 *
	 * @param string $id	The item unique id.
	 * @return Widget_ListLayout
	 */
	public function ListLayout($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'listlayout.class.php';
		return new Widget_ListLayout($id);
	}


	/**
	 * Constructs a Widget_VBoxLayout.
	 *
	 * @param string $id	The item unique id.
	 * @return Widget_VBoxLayout
	 */
	public function VBoxLayout($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'vboxlayout.class.php';
		return new Widget_VBoxLayout($id);
	}


	/**
	 * Constructs a Widget_HBoxLayout.
	 *
	 * @param string $id	The item unique id.
	 * @return Widget_HBoxLayout
	 */
	public function HBoxLayout($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'hboxlayout.class.php';
		return new Widget_HBoxLayout($id);
	}


	/**
	 * Constructs a Widget_FlowLayout.
	 *
	 * @param string $id	The item unique id.
	 * @return Widget_FlowLayout
	 */
	public function FlowLayout($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'flowlayout.class.php';
		return new Widget_FlowLayout($id);
	}


	/**
	 * Constructs a Widget_MulticolumnLayout.
	 *
	 * @param string $id	The item unique id.
	 * @return Widget_MulticolumnLayout
	 */
	public function MulticolumnLayout($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'multicolumnlayout.class.php';
		return new Widget_MulticolumnLayout($id);
	}


	/**
	 * Constructs a Widget_GridLayout.
	 *
	 * @param int		$nbColumns	The number of columns in the grid
	 * @param string	$id			The item unique id.
	 * @return Widget_GridLayout
	 */
	public function GridLayout($nbColumns = -1, $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'gridlayout.class.php';
		return new Widget_GridLayout($nbColumns, $id);
	}




	/**
	 * Constructs a Widget_Page.
	 *
	 * @param Widget_Layout	$layout		The layout that will manage how widgets are displayed in this container.
	 * @param string		$id			The item unique id.
	 * @return Widget_Page
	 */
	public function Page($id = null, Widget_Layout $layout = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'page.class.php';
		return new Widget_Page($id, $layout);
	}


	/**
	 * Constructs a Widget_BabPage.
	 *
	 * @param Widget_Layout	$layout		The layout that will manage how widgets are displayed in this container.
	 * @param string		$id			The item unique id.
	 * @return Widget_BabPage
	 */
	public function BabPage($id = null, Widget_Layout $layout = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'babpage.class.php';
		return new Widget_BabPage($id, $layout);
	}


	/**
	 * Constructs a Widget_Label.
	 *
	 * @param string	$text	The label text.
	 * @param string	$id		The item unique id.
	 * @return Widget_Label
	 */
	public function Label($text = '', $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'label.class.php';
		return new Widget_Label($text, $id);
	}


	/**
	 * Constructs a Widget_RichText.
	 *
	 * @param string	$text	The rich text.
	 * @param string	$id		The item unique id.
	 * @return Widget_RichText
	 */
	public function RichText($text = '', $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'richtext.class.php';
		return new Widget_RichText($text, $id);
	}


	/**
	 * Constructs a Widget_Html.
	 *
	 * @param string	$text	The html text.
	 * @param string	$id		The item unique id.
	 * @return Widget_Html
	 */
	public function Html($text = '', $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'html.class.php';
		return new Widget_Html($text, $id);
	}


	/**
	 * Constructs a Widget_SimpleHtmlEdit.
	 *
	 * @param string	$id		The item unique id.
	 * @return Widget_SimpleHtmlEdit
	 */
	public function SimpleHtmlEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'simplehtmledit.class.php';
		return new Widget_SimpleHtmlEdit($id);
	}


	/**
	 * Constructs a Widget_BabHtmlEdit.
	 *
	 * @param string	$id		The item unique id.
	 * @return Widget_BabHtmlEdit
	 */
	public function BabHtmlEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'babhtmledit.class.php';
		return new Widget_BabHtmlEdit($id);
	}


	/**
	 * Constructs a Widget_SimpleTreeView.
	 *
	 * @param string	$id		The item unique id.
	 * @return Widget_SimpleTreeView
	 */
	public function SimpleTreeView($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'simpletreeview.class.php';
		return new Widget_SimpleTreeView($id);
	}


	/**
	 * Constructs a Widget_Link.
	 *
	 * @param string	$text	The link text.
	 * @param string	$url	The link url or a Widget_Action.
	 * @param string	$id		The item unique id.
	 * @return Widget_Link
	 */
	public function Link($text = '', $url = '', $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'link.class.php';
		return new Widget_Link($text, $url, $id);
	}


	/**
	 * Constructs a Widget_Title.
	 *
	 * @param string | Widget_Displayable_Interface		$text	The title text.
	 * @param int										$level	The title level [1-6].
	 * @param string									$id		The item unique id.
	 * @return Widget_Title
	 */
	public function Title($text = '', $level = 2, $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'title.class.php';
		return new Widget_Title($text, $level, $id);
	}



	/**
	 * Constructs a Widget_Hidden.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Hidden
	 */
	public function Hidden($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'hidden.class.php';
		return new Widget_Hidden($id);
	}


	/**
	 * Constructs a Widget_LineEdit.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_LineEdit
	 */
	public function LineEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'lineedit.class.php';
		return new Widget_LineEdit($id);
	}


	/**
	 * Constructs a Widget_EmailLineEdit.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_EmailLineEdit
	 */
	public function EmailLineEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'emaillineedit.class.php';
		return new Widget_EmailLineEdit($id);
	}


	/**
	 * Constructs a Widget_UrlLineEdit.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_EmailLineEdit
	 */
	public function UrlLineEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'urllineedit.class.php';
		return new Widget_UrlLineEdit($id);
	}


	/**
	 * Constructs a Widget_TelLineEdit for telephon number
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_TelLineEdit
	 */
	public function TelLineEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'tellineedit.class.php';
		return new Widget_TelLineEdit($id);
	}


	/**
	 * Constructs a Widget_SearchLineEdit for telephon number
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_SearchLineEdit
	 */
	public function SearchLineEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'searchlineedit.class.php';
		return new Widget_SearchLineEdit($id);
	}


	//	/**
//	 * Constructs a Widget_NumberEdit.
//	 *
//	 * @param string		$id			The item unique id.
//	 * @return Widget_NumberEdit
//	 */
//	public function NumberEdit($id = null)
//	{
//		require_once FUNC_WIDGETS_PHP_PATH . 'numberedit.class.php';
//		return new Widget_NumberEdit($id);
//	}



	/**
	 * Constructs a Widget_RatingEdit.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_DateEdit
	 */
	public function RatingEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'ratingedit.class.php';
		return new Widget_RatingEdit($id);
	}



	/**
	 * Constructs a Widget_DateEdit.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_DateEdit
	 */
	public function DateEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'dateedit.class.php';
		return new Widget_DateEdit($id);
	}


	/**
	 * Constructs a Widget_TimeEdit.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_TimeEdit
	 */
	public function TimeEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'timeedit.class.php';
		return new Widget_TimeEdit($id);
	}


	/**
	 * Constructs a Widget_SuggestLineEdit.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_SuggestLineEdit
	 */
	public function SuggestLineEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'suggestlineedit.class.php';
		return new Widget_SuggestLineEdit($id);
	}


	/**
	 * Constructs a Widget_SuggestTextEdit.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_SuggestTextEdit
	 */
	public function SuggestTextEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'suggesttextedit.class.php';
		return new Widget_SuggestTextEdit($id);
	}


	/**
	 * Constructs a Widget_SuggestCountry. with country suggestions
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_SuggestCountry
	 */
	public function SuggestCountry($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'suggestcountry.class.php';
		return new Widget_SuggestCountry($id);

	}

	/**
	 * Constructs a Widget_SuggestPostalCode. with postal code suggestions
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_SuggestPostalCode
	 */
	public function SuggestPostalCode($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'suggestpostalcode.class.php';
		return new Widget_SuggestPostalCode($id);

	}

	/**
	 * Constructs a Widget_SuggestPlaceName. with city/town/village suggestions
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_SuggestPlaceName
	 */
	public function SuggestPlaceName($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'suggestplacename.class.php';
		return new Widget_SuggestPlaceName($id);

	}

	/**
	 * Constructs a Widget_SuggestUser. with id/name suggestions
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_SuggestUser
	 */
	public function SuggestUser($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'suggestuser.class.php';
		return new Widget_SuggestUser($id);

	}

	/**
	 * Constructs a Widget_SuggestGroup. with id/name suggestions
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_SuggestGroup
	 */
	public function SuggestGroup($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'suggestgroup.class.php';
		return new Widget_SuggestGroup($id);

	}


	/**
	 * Constructs a Widget_SuggestSelect.
	 * propose suggestion and confirm selection with a select box.
	 * a click on a suggestion grab the id of the object
	 * typed text of suggest line edit populate the select box to get the id on the next submit
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_SuggestSelect
	 */
	public function SuggestSelect($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'suggestselect.class.php';
		return new Widget_SuggestSelect($id);

	}


	/**
	 * Contstruct a Widget_DrilldownMenu
	 * to select a value into a tree
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_DrilldownMenu
	 */
	public function DrilldownMenu($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'drilldownmenu.class.php';
		return new Widget_DrilldownMenu($id);
	}



	/**
	 * Constructs a Widget_TextEdit.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_TextEdit
	 */
	public function TextEdit($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'textedit.class.php';
		return new Widget_TextEdit($id);
	}


	/**
	 * Constructs a Widget_Uploader.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Uploader
	 */
	public function Uploader($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'uploader.class.php';
		return new Widget_Uploader($id);
	}


	/**
	 * Constructs a Widget_MultiUploader.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_MultiUploader
	 */
	public function MultiUploader($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'multiuploader.class.php';
		return new Widget_MultiUploader($id);
	}


	/**
	 * Constructs a Widget_ColorPicker.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_ColorPicker
	 */
	public function ColorPicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'colorpicker.class.php';
		return new Widget_ColorPicker($id);
	}


	/**
	 * Constructs a Widget_FilePicker.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_FilePicker
	 */
	public function FilePicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'filepicker.class.php';
		return new Widget_FilePicker($id);
	}

	/**
	 * Constructs a Widget_ImagePicker.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_ImagePicker
	 */
	public function ImagePicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'imagepicker.class.php';
		return new Widget_ImagePicker($id);
	}

	/**
	 * Constructs a Widget_ImageCropper.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_ImageCropper
	 */
	public function ImageCropper($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'imagecropper.class.php';
		return new Widget_ImageCropper($id);
	}


	/**
	 * Constructs a Widget_Button.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_Button
	 */
	public function Button($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'button.class.php';
		return new Widget_Button($id);
	}


	/**
	 * Constructs a Widget_SubmitButton.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_SubmitButton
	 */
	public function SubmitButton($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'submitbutton.class.php';
		return new Widget_SubmitButton($id);
	}


	/**
	 * Constructs a Widget_ResetButton.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_ResetButton
	 */
	public function ResetButton($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'resetbutton.class.php';
		return new Widget_ResetButton($id);
	}


	/**
	 * Constructs a Widget_CheckBox.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_CheckBox
	 */
	public function CheckBox($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'checkbox.class.php';
		return new Widget_CheckBox($id);
	}


	/**
	 * Constructs a Widget_CheckBoxModelView.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_CheckBoxModelView
	 */
	public function CheckBoxModelView($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'checkboxmodelview.class.php';
		return new Widget_CheckBoxModelView($id);
	}


	/**
	 * Constructs a Widget_TimePicker.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_TimePicker
	 */
	public function TimePicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'timepicker.class.php';
		return new Widget_TimePicker($id);
	}


	/**
	 * Constructs a Widget_DatePicker.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_DatePicker
	 */
	public function DatePicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'datepicker.class.php';
		return new Widget_DatePicker($id);
	}


	/**
	 * Constructs a Widget_PeriodPicker.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_DatePicker
	 */
	public function PeriodPicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'periodpicker.class.php';
		return new Widget_PeriodPicker($id);
	}


	/**
	 * Constructs a Widget_UserPicker.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_UserPicker
	 */
	public function UserPicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'userpicker.class.php';
		return new Widget_UserPicker($id);
	}


	/**
	 * Constructs a Widget_GroupPicker.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_GroupPicker
	 */
	public function GroupPicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'grouppicker.class.php';
		return new Widget_GroupPicker($id);
	}


	/**
	 * Constructs a Widget_BabFilePicker.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_BabFilePicker
	 */
	public function BabFilePicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'babfilepicker.class.php';
		return new Widget_BabFilePicker($id);
	}
	
	
	/**
	 * Constructs a Widget_SitemapItemPicker.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_SitemapItemPicker
	 */
	public function SitemapItemPicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'sitemapitempicker.class.php';
		return new Widget_SitemapItemPicker($id);
	}
	


	/**
	 * Constructs a Widget_TopicPicker.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_TopicPicker
	 */
	public function TopicPicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'topicpicker.class.php';
		return new Widget_TopicPicker($id);
	}


	/**
	 * Constructs a Widget_TopicCategoryPicker.
	 *
	 * @param string $id			The item unique id.
	 * @return Widget_TopicCategoryPicker
	 */
	public function TopicCategoryPicker($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'topiccategorypicker.class.php';
		return new Widget_TopicCategoryPicker($id);
	}


	/**
	 * Constructs a Widget_ListView.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_ListView
	 */
	public function ListView($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'listview.class.php';
		return new Widget_ListView($id);
	}


	/**
	 * Constructs a Widget_TableView.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_TableView
	 */
	public function TableView($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'tableview.class.php';
		return new Widget_TableView(null, $id);
	}


	/**
	 * Constructs a Widget_TableModelView.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_TableModelView
	 */
	public function TableModelView($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'tablemodelview.class.php';
		return new Widget_TableModelView($id);
	}


	/**
	 * Constructs a Widget_PageSelector
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_PageSelector
	 */
	public function PageSelector($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'pageselector.class.php';
		return new Widget_PageSelector($id);
	}


	/**
	 * Constructs a Widget_Icon.
	 *
	 * @param string		$text	The label text.
	 * @param string		$image	The image name.
	 * @param string		$id		The item unique id.
	 * @return Widget_Icon
	 */
	public function Icon($text = '', $image = '', $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'icon.class.php';
		return new Widget_Icon($text, $image, $id);
	}

	/**
	 * Create a icon for a file
 	 * the file will have a default icon based on filetype and a thumbnail if possible
 	 * the thumbnail is loaded after page load
 	 *
 	 * @see Widget_Icon
 	 * @see Widget_ImageThumbnail
 	 *
 	 * @param 	string		$text		The label text.
 	 * @param	bab_Path 	$path		path to file
 	 * @param	string		$id			The item unique id.
	 */
	public function FileIcon($text = '', bab_Path $path = null, $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'fileicon.class.php';
		return new Widget_FileIcon($text, $path, $id);
	}


	/**
	 * Constructs a Widget_Image.
	 *
	 * @param string		$imageSrc	The image source (url).
	 * @param string		$labelText	The label text.
	 * @param string		$id			The item unique id.
	 *
	 * @return Widget_Image
	 */
	public function Image($imageSrc = '', $labelText = '', $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'image.class.php';
		return new Widget_Image($imageSrc, $labelText, $id);
	}


	/**
	 * Constructs a Widget_Image of a country flag
	 *
	 * @param string		$isoCode	2 char
	 * @param string		$id			The item unique id.
	 *
	 * @return Widget_Image | null
	 */
	public function CountryFlag($isoCode, $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'image.class.php';
		$imageSrc = bab_getAddonInfosInstance('widgets')->getImagesPath().'16x11_country/'.strtolower($isoCode).'.gif';
		if (!file_exists($imageSrc))
		{
			return null;
		}

		return new Widget_Image($imageSrc, '', $id);
	}


	/**
	 * Create thumbnail for a file
	 * the file will have a default image and a thumbnail if possible
	 * the thumbnail is loaded after page load
	 *
	 * @see Widget_Image
	 * @see Widget_FileIcon
	 *
	 * @param	bab_Path 	$path		path to file
	 * @param	string		$id			The item unique id.
	 */
	public function ImageThumbnail(bab_Path $imageSrc, $labelText = '', $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'imagethumbnail.class.php';
		return new Widget_ImageThumbnail($imageSrc, $labelText, $id);
	}


	/**
	 * Constructs a Widget_ImageZoomer.
	 * display a small cliquable image, the big image is displayed in a overlay frame when the thumbnail is clicked
	 *
	 *
	 * @param string		$imageSrc	The image source (url).
	 * @param string		$labelText	The label text.
	 * @param string		$id			The item unique id.
	 * @return Widget_Image
	 */
	public function ImageZoomer($smallImageSrc = '', $bigImageSrc = '', $labelText = '', $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'imagezoomer.class.php';
		return new Widget_ImageZoomer($smallImageSrc, $bigImageSrc, $labelText, $id);
	}

	/**
	 * Constructs a Widget_ImageZoomerThumbnail.
	 * display a small cliquable image, the big image is displayed in a overlay frame when the thumbnail is clicked
	 *
	 * @param bab_Path		$imagePath	The image source (path).
	 * @param string		$labelText	The label text.
	 * @param string		$id			The item unique id.
	 * @return Widget_Image
	 */
	public function ImageZoomerThumbnail(bab_Path $imagePath, $labelText = '', $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'imagezoomerthumbnail.class.php';
		return new Widget_ImageZoomerThumbnail($imagePath, $labelText, $id);
	}


	/**
	 * Constructs a Widget_Frame.
	 *
	 * @param Widget_Layout	$layout		The layout that will manage how widgets are displayed in this container.
	 * @param string		$id			The item unique id.
	 * @return Widget_Frame
	 */
	public function Frame($id = null, Widget_Layout $layout = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'frame.class.php';
		return new Widget_Frame($id, $layout);
	}


	/**
	 * Constructs a Widget_NamedContainer.
	 * A Widget_NamedContainer is a container not displayed on page but with a name and optional metadata
	 *
	 * @param	string	$name		The item name
	 * @param 	string	$id			The item unique id.
	 * @return Widget_NamedContainer
	 */
	public function NamedContainer($name = null, $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'namedcontainer.class.php';
		return new Widget_NamedContainer($name, $id);
	}


	/**
	 * Constructs a Widget_Menu.
	 *
	 * @param Widget_Layout	$layout		The layout that will manage how widgets are displayed in this container.
	 * @param string		$id			The item unique id.
	 * @return Widget_Menu
	 */
	public function Menu($id = null, Widget_Layout $layout = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'menu.class.php';
		return new Widget_Menu($id, $layout);
	}


	/**
	 * Constructs a Widget_Form.
	 *
	 * @param Widget_Layout	$layout		The layout that will manage how widgets are displayed in this container.
	 * @param string		$id			The item unique id.
	 * @return Widget_Form
	 */
	public function Form($id = null, Widget_Layout $layout = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'form.class.php';
		return new Widget_Form($id, $layout);
	}


	/**
	 * Constructs a Widget_Form with embeded fields to manage CSV importation.
	 *
	 * @param Widget_Layout	$layout		The layout that will manage how widgets are displayed in this container.
	 * @param string		$id			The item unique id.
	 * @return Widget_CsvImportForm
	 */
	public function CsvImportForm($id = null, Widget_Layout $layout = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'csvimportform.class.php';
		return new Widget_CsvImportForm($id, $layout);
	}


	/**
	 * Constructs a Widget_MultiField.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_MultiField
	 */
	public function MultiField($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'multifield.class.php';
		return new Widget_MultiField($id);
	}


	/**
	 * Constructs a Widget_Select.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Select
	 */
	public function Select($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'select.class.php';
		return new Widget_Select($id);
	}


	/**
	 * Constructs a Widget_Select.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Select
	 */
	public function Multiselect($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'multiselect.class.php';
		return new Widget_Multiselect($id);
	}


	/**
	 * Constructs a Widget_RadioSet.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_RadioSet
	 */
	public function RadioSet($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'radioset.class.php';
		return new Widget_RadioSet($id);
	}


	/**
	 * Constructs a Widget_RadioMenu.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_RadioMenu
	 */
	public function RadioMenu($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'radiomenu.class.php';
		return new Widget_RadioMenu($id);
	}


	/**
	 * Constructs a Widget_Radio.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Radio
	 */
	public function Radio($id = null, $radioSet = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'radio.class.php';
		return new Widget_Radio($id, $radioSet);
	}


	/**
	 * Constructs a Widget_Tabs.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Tabs
	 */
	public function Tabs($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'tabs.class.php';
		return new Widget_Tabs($id);
	}


	/**
	 * Constructs a Widget_Tab.
	 *
	 * @param 	string			$id			The item unique id.
	 * @param	string			$label
	 * @param	Widget_Layout	$layout
	 * @return Widget_Tab
	 */
	public function Tab($id = null, $label = null, $layout = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'tab.class.php';
		return new Widget_Tab($id, $label, $layout);
	}


	/**
	 * Constructs a Widget_Accordions.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Accordions
	 */
	public function Accordions($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'accordions.class.php';
		return new Widget_Accordions($id);
	}


	/**
	 * Constructs a Widget_Calendar.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Calendar
	 */
	public function Calendar($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'calendar.class.php';
		return new Widget_Calendar($id);
	}

	/**
	 * Constructs a Widget_MobileMonthCalendar.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_MobileMonthCalendar
	 */
	public function MobileMonthCalendar($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'calendar.class.php';
		return new Widget_MobileMonthCalendar($id);
	}


	/**
	 * Constructs a Widget_FullCalendar.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_FullCalendar
	 */
	public function FullCalendar($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'fullcalendar.class.php';
		return new Widget_FullCalendar($id);
	}


	/**
	 * Constructs a Widget_Timeline.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Timeline
	 */
	public function Timeline($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'timeline.class.php';
		return new Widget_Timeline($id);
	}


	/**
	 * Constructs a Widget_Filter.
	 *
	 * @param string		$id			The item unique id.
	 * @return Widget_Filter
	 */
	public function Filter($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'filter.class.php';
		return new Widget_Filter($id);
	}


	/**
	 * Constructs a Widget_HtmlCanvas.
	 *
	 * @return Widget_HtmlCanvas
	 */
	public function HtmlCanvas()
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'htmlcanvas.class.php';
		return new Widget_HtmlCanvas;
//		return new Widget_Html5Canvas;
	}


	/**
	 * Construct a Widget_Action
	 *
	 * @return Widget_Action
	 */
	public function Action()
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'action.class.php';
		return new Widget_Action;
	}

	/**
	 * Set items in parameters as a default Layout
	 * @return Widget_Layout
	 */
	public function Items() {
		$layout = $this->Layout();
		foreach(func_get_args() as $item) {
			if (is_null($item)) {
				continue;
			}
			$layout->addItem($item);
		}

		return $layout;
	}


	/**
	 * Set items in parameters as a HBoxLayout
	 * @return Widget_HBoxLayout
	 */
	public function HBoxItems() {
		$layout = $this->HBoxLayout();
		foreach(func_get_args() as $item) {
			if (is_null($item)) {
				continue;
			}
			$layout->addItem($item);
		}

		return $layout;
	}


	/**
	 * Set items in parameters as a VBoxLayout
	 * @return Widget_VBoxLayout
	 */
	public function VBoxItems() {
		$layout = $this->VBoxLayout();
		foreach(func_get_args() as $item) {
			if (is_null($item)) {
				continue;
			}
			$layout->addItem($item);
		}

		return $layout;
	}


	/**
	 * Set items in parameters as a FlowLayout
	 * @return Widget_FlowLayout
	 */
	public function FlowItems() {
		$layout = $this->FlowLayout();
		foreach(func_get_args() as $item) {
			if (is_null($item)) {
				continue;
			}
			$layout->addItem($item);
		}

		return $layout;
	}


	/**
	 * construct a pair widget
	 * @return Widget_Pair
	 */
	public function Pair($first, $second) {
		return $this->create('Widget_Pair')
			->setFirst($first)
			->setSecond($second);
	}


	/**
	 * Gauge
	 *
	 * @param string $id
	 * @return Widget_Gauge
	 */
	public function Gauge($id = null) {
		require_once FUNC_WIDGETS_PHP_PATH . 'gauge.class.php';
		return new Widget_Gauge($id);
	}

	/**
	 * Progress bar
	 *
	 * @param string $id
	 * @return Widget_ProgressBar
	 */
	public function ProgressBar($id = null) {
		require_once FUNC_WIDGETS_PHP_PATH . 'progressbar.class.php';
		return new Widget_ProgressBar($id);
	}


	/**
	 * Section
	 *
	 * @param string									$headerText			The section header
	 * @param Widget_Layout	| Widget_ContainerWidget	$containerItem		The section content layout
	 * @param int										$level				Similar to Title level
	 * @param string 									$id
	 * @return Widget_Section
	 */
	public function Section($headerText, $containerItem = null, $level = 3, $id = null) {
		require_once FUNC_WIDGETS_PHP_PATH . 'section.class.php';
		return new Widget_Section($headerText, $containerItem, $level, $id);
	}


	/**
	 * ACL input widget
	 * @param string $id
	 * @return Widget_Acl
	 */
	public function Acl($id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'acl.class.php';
		return new Widget_Acl($id);
	}


	/**
	 * Dockable
	 *
	 * @param 	string			[$label]	The dock button label.
	 * @param	Widget_Action	[$action]	The dock content url
	 * @param 	string			[$id]		The item unique id.
	 *
	 * @return Widget_Acl
	 */
	public function Dockable($label = null, Widget_Action $action = null, $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'dockable.class.php';
		return new Widget_Dockable($label, $action, $id);
	}


	/**
	 * Constructs a Widget_DelayedItem.
	 *
	 * @param Widget_Action $delayedAction  A widget action returning a Widget_DisplayableInterface
	 * @param string        $id             The item unique id.
	 * @return Widget_DelayedItem
	 */
	public function DelayedItem(Widget_Action $delayedAction = null, $id = null)
	{
		require_once FUNC_WIDGETS_PHP_PATH . 'delayeditem.class.php';
		return new Widget_DelayedItem($delayedAction, $id);
	}



	/**
	 * Call before page creation
	 * @return mixed
	 */
	public static function onBeforePageCreated()
	{

		if (isset($_SESSION['addon_widgets'])) {

			$requested_uid = bab_rp('widget_filepicker_job_uid');
			// uid can be an array of job uid to do (submit of full form) or one uid (ajax)

			if (!is_array($requested_uid)) {
				$requested_uid = array($requested_uid);
			}

			foreach($_SESSION['addon_widgets'] as $widget_class => $jobs) {

				foreach($jobs as $uid => $job) {

					if (!in_array($uid, $requested_uid)) {

						//unset($_SESSION['addon_widgets'][$widget_class][$uid]);
						continue;
					}

					if (isset($job['method'])) {
						if (isset($job['params'])) {
							$params = $job['params'];
						} else {
							$params = array();
						}



						self::includePhpClass($widget_class);
						call_user_func_array(array($widget_class, $job['method']), $params);
					}
				}

			}


		}
	}


	/**
	 * This method can be used to store key/values for the currently
	 * connected user. They can be retrieved later using  Func_Widgets::getUserConfiguration().
	 *
	 * @see Func_Widgets::getUserConfiguration()
	 *
	 * @param string $key   A key in the form of a path (eg. 'tableview_a/visiblecolumns/name').
	 * @param string $value The value to store.
	 * @param string $addon
	 */
	public function setUserConfiguration($key, $value, $addon = 'widgets')
	{
		if (!bab_isUserLogged()) {
			$_SESSION['/' . $addon . '/' . $key] = $value;
			return;
		}
		$registry = bab_getRegistryInstance();
		$registry->changeDirectory('/' . $addon);
		$registry->changeDirectory('user');
		$registry->changeDirectory($GLOBALS['BAB_SESS_USERID']);
		$keyPath = explode('/', $key);
		$key = array_pop($keyPath);
		$path = implode('/', $keyPath);
		$registry->changeDirectory($path);

		$registry->setKeyValue($key, $value);
	}




	/**
	 * This method can be used to retrieve stored configuration key/values for the currently
	 * connected user.
	 *
	 * @see Func_Widgets::setUserConfiguration()
	 *
	 * @param string $key   A key in the form of a path (eg. 'tableview_a/visiblecolumns/name').
	 * @param string $addon
	 */
	public function getUserConfiguration($key, $addon = 'widgets')
	{
		if (!bab_isUserLogged()) {
			if (isset($_SESSION['/' . $addon . '/' . $key])) {
				return $_SESSION['/' . $addon . '/' . $key];
			}
			return null;
		}
		$registry = bab_getRegistryInstance();
		$registry->changeDirectory('/' . $addon);
		$registry->changeDirectory('user');
		$registry->changeDirectory($GLOBALS['BAB_SESS_USERID']);
		$keyPath = explode('/', $key);
		$key = array_pop($keyPath);
		$path = implode('/', $keyPath);
		$registry->changeDirectory($path);
		return $registry->getValue($key);
	}




	/**
     * Deletes all the user configuration key/values under the specified path.
     *
     * @param string $path   The path to delete (eg. 'tableview_a/visiblecolumns').
	 */
	public function deleteUserConfiguration($path, $addon = 'widgets')
	{
		if (!bab_isUserLogged()) {
			unset($_SESSION['/' . $addon . '/' . $path]);
		}
		$registry = bab_getRegistryInstance();
		$registry->changeDirectory('/' . $addon);
		$registry->changeDirectory('user');
		$registry->changeDirectory($GLOBALS['BAB_SESS_USERID']);
		$registry->changeDirectory($path);
		return $registry->deleteDirectory();
	}




	/**
	 * Returns user agent information.
	 *
	 * @return uagent_info
	 */
	public function getUserAgentInfo()
	{
		require_once dirname(__FILE__) . '/mdetect.php';

		$userAgentInfo = bab_getInstance('uagent_info');
		return $userAgentInfo;
	}


}



/**
 * Returns the specified string translated into the current user's language.
 *
 * @param string		$str
 * @return string
 */
function widget_translate($str)
{
	$translation = bab_translate($str, 'widgets');

	return $translation;
}
