<?php
//-------------------------------------------------------------------------
// OVIDENTIA http://www.ovidentia.org
// Ovidentia is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.
//-------------------------------------------------------------------------
/**
 * @license http://opensource.org/licenses/gpl-license.php GNU General Public License (GPL)
 * @copyright Copyright (c) 2008 by CANTICO ({@link http://www.cantico.fr})
 */
include_once 'base.php';





/**
 * The Canvas class provides primitives that must be used by widgets to generate their output.
 * @abstract
 */
class Widget_Canvas
{

	/**
	 * Returns a new Widget_CanvasOptions.
	 *
	 * @return Widget_CanvasOptions
	 */
	static public function Options()
	{
		require_once dirname(__FILE__).'/canvasoptions.class.php';
		return new Widget_CanvasOptions();
	}

	/**
	 * Returns a text element.
	 *
	 * @param string $text
	 * @return mixed
	 * @abstract
	 */
	public function text($text) { }

	/**
	 * Returns a canvas label element.
	 *
	 * @param string				$id				The unique id (for the page).
	 * @param array					$classes		An array containing the classes as returned by Widget_Item::getClasses()
	 * @param string				$text
	 * @param string				$forId			The id of the element this label is for.
	 * @param Widget_CanvasOptions	$options
	 * @param string				$title			The (optional) tooltip
	 * @return mixed
	 * @abstract
	 */
	public function label($id, $classes, $text, $forId, $options = null, $style = null) { }


	/**
	 * Returns a canvas richtext element.
	 *
	 * @param string	$id					The unique id (for the page).
	 * @param array		$classes			An array containing the classes as returned by Widget_Item::getClasses()
	 * @param string	$text
	 * @param int		$renderingOption	rendering options BAB_HTML_*.
	 * @param Widget_CanvasOptions	$options
	 * @return mixed
	 * @abstract
	 */
	public function richtext($id, $classes, $text, $renderingOption, $options) { }



	/**
	 * Returns a canvas div element.
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param mixed					$displayableItems		Array of Widget_Displayable_Interface objects or strings.
	 * @param Widget_CanvasOptions	$options
	 * @param string				$title					The optional tooltip
	 * @return mixed
	 * @abstract
	 */
	public function div($id, $classes, $displayableItems, $options = null, $title = null) { }


	/**
	 * Returns a canvas header element.
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects or strings
	 * @param int					$level					integer from 1 to 6, represent the importance of the header
	 * @param Widget_CanvasOptions	$options				Canvas options
	 * @param string				$title					The optional tooltip
	 * @return string
	 */
	public function h($id, $classes, $displayableItems, $level, $options = null, $title = null) { }


	/**
	 * Returns a canvas span element.
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects.
	 * @param Widget_CanvasOptions	$options
	 * @param string				$title					The optional tooltip
	 * @return string
	 */
	public function span($id, $classes, $displayableItems, $options = null, $title = null) { }





	/**
	 * Display an image
	 * @param string				$id				The unique id (for the page).
	 * @param array					$classes		An array containing the classes as returned by Widget_Item::getClasses().
	 * @param string				$text			The alternate text if the image cannot be displayed.
	 * @param string				$url			The url of the image.
	 * @param Widget_CanvasOptions	$options
	 * @param string				$title			The optional tooltip
	 * @return 	string
	 * @abstract
	 */
	public function image($id, $classes, $text, $url, $options = null, $title = null) { }


	/**
	 * Display a link.
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects or strings.
	 * @param string				$url
	 * @param Widget_CanvasOptions	$options
	 * @param string				$title					The optional tooltip
	 * @param array					$attributes
	 * @return string
	 */
	public function linkContainer($id, $classes, $displayableItems, $url, $options = null, $title = null, $attributes = null) { }


	/**
	 * Display a form
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses()
	 * @param boolean				$readonly				Form generate a modification of data on server if true
	 * @param array					$hiddenFields			Hidden fields
	 * @param array					$displayableItems
	 * @param boolean				$selfPageHiddenFields	Add necessary hidden fields in form to submit on the same page
	 * @param string				$anchor					Set anchor in destination page
	 * @param Widget_CanvasOptions	$options				Canvas options
	 * @param string				$title					The optional tooltip
	 * @param array					$attributes
	 *
	 * @return mixed
	 * @abstract
	 */
	public function form($id, $classes, $readonly, $hiddenFields, $displayableItems, $selfPageHiddenFields, $anchor = null, $options = null, $title = null, $attributes = null) { }



	/**
	 * Returns an checkbox input field.
	 *
	 * @param string	$id					The unique id (for the page).
	 * @param array		$classes			An array containing the classes as returned by Widget_Item::getClasses()
	 * @param array		$fullName			An array containing the full name as returned by Widget_Item::getFullName()
	 * @param bool		$value				The initial value.
	 * @param string	$checked_value
	 * @param string	$unchecked_value
	 * @param bool		$disabled
	 * @param string	$title				The tooltip title
	 * @return mixed
	 * @abstract
	 */
	public function checkBoxInput($id, $classes, $fullName, $value, $checked_value, $unchecked_value, $disabled, $title) { }




	/**
	 * Returns a button
	 *
	 * @param string				$id					The unique id (for the page).
	 * @param array					$classes			An array containing the classes as returned by Widget_Item::getClasses()
	 * @param bool					$disabled
	 * @param array					$displayableItems	Array of Widget_Displayable_Interface objects.
	 * @param Widget_CanvasOptions	$options			Canvas options
	 * @param string				$title				button title
	 * @return mixed
	 * @abstract
	 */
	public function button($id, $classes, $disabled, $displayableItems, $options, $title) { }




	/**
	 * Returns a single line text input field.
	 *
	 * @param string	$id					The unique id (for the page).
	 * @param array		$classes			An array containing the classes as returned by Widget_Item::getClasses()
	 * @param array		$fullName			An array containing the full name as returned by Widget_Item::getFullName()
	 * @param string	$value				The initial value.
	 * @param int		$size				The size in character or NULL.
	 * @param int		$maxSize			The maximum size in characters or NULL.
	 * @param bool		$disabled
	 * @param string	$title				A title string that will appear as a tooltip.
	 * @param bool		$readOnly			The value of the readonly parameter.
	 * @return mixed
	 * @abstract
	 */
	public function lineInput($id, $classes, $fullName, $value, $size, $maxSize, $disabled, $title, $readOnly = 0, $autoComplete = 1, $type = 'text') { }

	/**
	 * Returns a multi-line text input field.
	 *
	 * @param string	$id					The unique id (for the page).
	 * @param array		$classes			An array containing the classes as returned by Widget_Item::getClasses()
	 * @param array		$fullName			An array containing the full name as returned by Widget_Item::getFullName()
	 * @param string	$value				The initial value.
	 * @param int		$width				The width in character or NULL.
	 * @param int		$height				The height in characters or NULL.
	 * @param bool		$disabled
	 * @return string
	 */
	public function textInput($id, $classes, $fullName, $value, $width, $height, $disabled) { }

	/**
	 * Returns a file uploader.
	 *
	 * @param string	$id					The unique id (for the page).
	 * @param array		$classes			An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array		$fullName			An array containing the full name as returned by Widget_Item::getFullName().
	 * @param array		$acceptedMimeType   The mime type.
	 * @param int		$size				The size of uploader.
	 * @param bool		$multiple			Multiple file upload, default false
	 * @return mixed
	 * @abstract
	 */
	public function fileUpload($id, $classes, $fullName, $acceptedMimeType, $size, $multiple) { }


	/**
	 * Returns an HTML select.
	 *
	 * @param string				$id					The unique (for the page) html id.
	 * @param array					$classes			An array containing the classes as returned by Widget_Item::getClasses()
	 * @param array					$fullName			An array containing the full name as returned by Widget_Item::getFullName()
	 * @param mixed					$value				The initial value.
	 * @param array					$options			key values for select
	 * @param array					$optgroups			options in groups
	 * @param bool					$disabled
	 * @param bool					$size				number of lign display
	 * @param Widget_CanvasOptions	$canvasOptions		Canvas options
	 * @param array					$classesOptions		Options classes
	 * @param array					$classesOptGroups	Options groups classes
	 * @param bool					$multiple			Multiple select
	 * @return string
	 * @abstract
	 */
	public function select($id, $classes, $fullName, $value, $options = null, $optgroups = null, $disabled = false, $size = null, $canvasOptions = null, $classesOptions = null, $classesOptGroups = null, $multiple = false)
	{
	}


	/**
	 * Return a default layout
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects.
	 * @param Widget_CanvasOptions	$options				Options for layout items
	 * @return string
	 */
	public function defaultLayout($id, $classes, $displayableItems, $options) {
		$content = '';
		foreach ($displayableItems as $item) {
			//if ($options) {
			//	$item->setCanvasOptions($options);
			//}
			$content .= $item->display($this);
		}
		return $content;
	}

	/**
	 * Returns canvas flow layout.
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects or string.
	 * @param Widget_CanvasOptions	$options				Options for layout items
	 * @return string
	 */
	public function flow($id, $classes, $displayableItems, $options = null)
	{
	}

	/**
	 * Returns canvas hbox layout.
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects or string.
	 * @param Widget_CanvasOptions	$options				Options for layout items
	 * @return string
	 */
	public function hbox($id, $classes, $displayableItems, $options = null) { }

	/**
	 * Returns canvas vbox layout.
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects or string.
	 * @param Widget_CanvasOptions	$options				Options for layout items
	 * @return string
	 */
	public function vbox($id, $classes, $displayableItems, $options = null, $role = null) { }


	/**
	 * Returns canvas grid layout.
	 *
	 * Experimental
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$rows					An array of strings returned by gridRow() @see Widget_Canvas::gridRow().
	 * @param array					$colOptions				An array of Widget_CanvasOptions that will be applied to corresponding columns.
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects or string.
	 * @param Widget_CanvasOptions	$options				Options for layout items.
	 * @return string
	 */
	public function grid($id, $classes, $rows = array(), $colOptions = array(), $options = null) { }


	/**
	 * Returns a grid row for the canvas grid layout.
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$cells					Array of Widget_Displayable_Interface objects or string.
	 * @param Widget_CanvasOptions	$options				Options for layout items
	 * @return string
	 */
	public function gridRow($id, $classes, $cells = array(), $options = null) { }


	/**
	 * Returns a grid cell for grid row the canvas grid layout.
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses().
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects or string.
	 * @param Widget_CanvasOptions	$options				Options for layout items
	 * @return string
	 */
	public function gridRowCell($id, $classes, $displayableItems = array(), $options = null, $rowSpan = null, $colSpan = null) { }


	/**
	 * Display a menu with tab items
	 *
	 * @param string				$id						The unique id (for the page).
	 * @param array					$classes				An array containing the classes as returned by Widget_Item::getClasses()
	 * @param array					$displayableItems		Array of Widget_Displayable_Interface objects.
	 * @param Widget_CanvasOptions	$options
	 *
	 * @return string
	 */
	public function tabs($id, $classes, $displayableItems, $options) { }




	/**
	 * Create anchor
	 * @param	string	$name
	 * @return string
	 */
	public function anchor($name) { }



	/**
	 * Create a treeview node.
	 *
	 * @param string	$id						The unique id (for the page).
	 * @param array		$classes				An array containing the classes as returned by Widget_Item::getClasses()
	 * @param array		$displayableItems		Array of Widget_Displayable_Interface objects.
	 * @param array		$childNodes				An array of Widget_TreeViewNode
	 * @param array		$options
	 * @return string
	 */
	public function treeviewnode($id, $classes, $displayableItems, $childNodes, $options = null) { }




	/**
	 * Create metadata container
	 * @param	string	$name
	 * @param	array	$metadata
	 * @return string
	 */
	public function metadata($name, $metadata) { }

	/**
	 * Load script
	 * @param	string	$widget_id
	 * @param	string	$filename
	 * @return string
	 */
	public function loadScript($widget_id, $filename) { }

	
	/**
	 * Load CSS style sheet
	 * @param	string	$widget_id
	 * @param	string	$filename
	 * @return string
	 */
	public function loadStyleSheet($filename) { }

	/**
	 * Returns an iframe element.
	 *
	 * @param string				$id			The unique id (for the page).
	 * @param string				$url		url of page to load in iframe
	 * @param Widget_CanvasOptions	$options
	 * @return mixed
	 * @abstract
	 */
	function iframe($id, $url, $options) { }

	/**
	 * Add optional decorations on page
	 * @param	Widget_Page	$page
	 * @param	array		$classes	class names used in widgets of the page
	 */
	public function setPageDecorations(Widget_Page $page, $classes) { }






}

