<?php
//-------------------------------------------------------------------------
// OVIDENTIA http://www.ovidentia.org
// Ovidentia is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.
//-------------------------------------------------------------------------
/**
 * @license http://opensource.org/licenses/gpl-license.php GNU General Public License (GPL)
 * @copyright Copyright (c) 2006 by CANTICO ({@link http://www.cantico.fr})
 */

require_once dirname(__FILE__) . '/containerwidget.class.php';
require_once dirname(__FILE__) . '/gridlayout.class.php';



/**
 * Constructs a Widget_TableView.
 *
 * @param string $id The item unique id.
 *
 * @return Widget_TableView
 */
function Widget_TableView($id = null)
{
	return new Widget_TableView($id);
}



/**
 * A Widget_TableView is a widget that displays a list of
 */
class Widget_TableView extends Widget_ContainerWidget implements Widget_Displayable_Interface
{
	/**
	 * @var array
	 */
	protected $_columnClasses;

	/**
	 * @var array
	 */
	protected $_rowClasses;

	/**
	 *
	 * @var string
	 */
	private $viewType;

	const VIEW_ICONS = 'widget-view-as-icon';
	const VIEW_LIST = 'widget-view-as-list';
	const VIEW_DETAILS = 'widget-view-as-details';
	const VIEW_CHART = 'widget-view-as-chart';

	/**
	 * @param array  $columns Optional columns classes
	 * @param string $id      The item unique id.
	 *
	 * @return Widget_TableView
	 */
	public function __construct($columns = null, $id = null)
	{
		parent::__construct($id);
		$this->_columnClasses = $columns;
		$layout = new Widget_GridLayout();
		$layout->setTableView($this);
		$this->setLayout($layout);
	}




	/**
	 * Creates a new section in the table view.
	 *
	 * @see Widget_GridLayout::addSection()
	 *
	 * @param string $id        The unique identifier (for the layout) of the new section.
	 * @param string $label     An optional text label.
	 * @param string $className An optional class name.
	 *
	 * @return Widget_TableView
	 */
	public function addSection($id = null, $label = null, $class = null)
	{
		$this->getLayout()->addSection($id, $label, $class);
		return $this;
	}

	/**
	 * Sets the current section to which subsquent addItem call will add items.
	 *
	 * @see Widget_TableView::addSection()
	 *
	 * @param string $id The identifier of the section (previously added by Widget_TableView::addSection()).
	 *
	 * @return Widget_TableView
	 */
	public function setCurrentSection($id)
	{
		$this->getLayout()->setCurrentSection($id);
		return $this;
	}

	/**
	 * Sets how items are displayed in the list view.
	 *
	 * Possible values are:
	 *  - Widget_TableView::VIEW_ICONS
	 *  - Widget_TableView::VIEW_LIST
	 *  - Widget_TableView::VIEW_DETAILS
	 *  - Widget_TableView::VIEW_CHART
	 *
	 * @param string $viewType
	 * @return Widget_TableView
	 */
	public function setView($viewType)
	{
		$this->viewType = $viewType;
		return $this;
	}



	/**
	 * Returns the type of list view (VIEW_ICONS, VIEW_LIST, VIEW_DETAILS).
	 *
	 * @see Widget_TableView::setView
	 * @return string
	 */
	public function getView()
	{
		return $this->viewType;
	}

	/**
	 * Get cell content or null if empty
	 * @return Widget_Item
	 */
	public function getCell($row, $col)
	{
		$layout = $this->getLayout();
		return $layout->getCell($row, $col);
	}


	/**
	 *
	 * @param int		$columnId
	 * @param string	$className
	 * @return Widget_TableView
	 */
	public function addColumnClass($columnId, $className)
	{
		$this->_columnClasses[$columnId][$className] = $className;
		return $this;
	}

	/**
	 * Get column classes before drawing the cell
	 * @param 	int 								$columnId
	 * @param	Widget_Displayable_Interface		$cellContent
	 * @return array
	 */
	public function getColumnClasses($columnId, Widget_Displayable_Interface $cellContent = null)
	{
		$classes = array();

		if (isset($this->_columnClasses[$columnId])) {
			$classes += $this->_columnClasses[$columnId];
		}

		return $classes;
	}

	/**
	 * Get row classes before drawing the row
	 * @param int $rowId
	 * @return Widget_TableView
	 */
	public function getRowClasses($rowId)
	{
		static $oddRow = true;
		$oddRow = !$oddRow;

		$classes = array('widget-table-row', $oddRow ? 'odd' : 'even');
		if (isset($this->_rowClasses[$rowId])) {
			$classes += $this->_rowClasses[$rowId];
		}

		return $classes;
	}


	/**
	 * Add a class on a row
	 * @param int		$rowId
	 * @param string	$className
	 * @return Widget_TableView
	 */
	public function addRowClass($rowId, $className)
	{
		$this->_rowClasses[$rowId][$className] = $className;
		return $this;
	}


	function getClasses() {
		$classes = parent::getClasses();
		$classes[] = 'widget-tableview';
		if ($this->viewType == Widget_TableView::VIEW_CHART) {
			$classes[] = 'widget-view-as-chart';
		}
		return $classes;
	}



	public function display(Widget_Canvas $canvas)
	{
		$layout = $this->getLayout();

		$display = $canvas->div(
			$this->getId(),
			$this->getClasses(),
			array($layout)
		)
		. $canvas->metadata($this->getId(), $this->getMetadata())
		. $canvas->loadScript($this->getId(), bab_getAddonInfosInstance('widgets')->getTemplatePath().'widgets.tableview.jquery.js');

// 		if ($this->viewType == Widget_TableView::VIEW_CHART) {
// 			$display .= $canvas->loadScript($this->getId(), bab_getAddonInfosInstance('widgets')->getTemplatePath().'widgets.graph.jquery.js');
// 		}

		return $display;
	}

	/**
	 * For debugging purpose.
	 *
	 * @param string $prefix
	 * @return string
	 * @ignore
	 */
	public function dump($prefix = '')
	{
		$dumpString = parent::dump($prefix);
		$dumpString .= $this->getLayout()->dump($prefix, false);
		return $dumpString;
	}
}

require_once dirname(__FILE__) . '/tablemodelview.class.php';
