<?php
/* Copyright (C) 2007-2009 Laurent Destailleur  <eldy@users.sourceforge.net>
 * Copyright (C) 2009-2010 Regis Houssin        <regis@dolibarr.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
 *  \file       htdocs/core/class/menubase.class.php
 *  \ingroup    core
 *  \version	$Id: menubase.class.php,v 1.14 2010/12/17 08:52:52 hregis Exp $
 *  \brief      File of class to manage dynamic menu entries
 *  \remarks	Initialy built by build_class_from_table on 2008-01-12 14:19
 */


/**
 *  \class      Menubase
 *  \brief      Class to manage menu entries
 */
class Menubase
{
	var $db;							// To store db handler
	var $error;							// To return error code (or message)
	var $errors=array();				// To return several error codes (or messages)

	var $id;

	var $menu_handler;
	var $module;
	var $type;
	var $mainmenu;
	var $fk_menu;
	var $position;
	var $url;
	var $target;
	var $titre;
	var $langs;
	var $level;
	var $leftmenu;		//<! 0=Left menu in pre.inc.php files must not be overwrite by database menu, 1=Must be
	var $perms;
	var $enabled;
	var $user;
	var $tms;


	/**
	 *      Constructor
	 *      @param     DB      Database handler
	 *      @param     menu_handler
	 *      @param     type
	 */
	function Menubase($DB,$menu_handler='',$type='')
	{
		$this->db = $DB;
		$this->menu_handler = $menu_handler;
		$this->type = $type;
		return 1;
	}


	/**
	 *      \brief      Create menu entry into database
	 *      \param      user        User that create
	 *      \return     int         <0 si ko, >0 si ok
	 */
	function create($user)
	{
		global $conf, $langs;

		// Clean parameters
		$this->menu_handler=trim($this->menu_handler);
		$this->module=trim($this->module);
		$this->type=trim($this->type);
		$this->mainmenu=trim($this->mainmenu);
		$this->fk_menu=trim($this->fk_menu);
		$this->position=trim($this->position);
		$this->url=trim($this->url);
		$this->target=trim($this->target);
		$this->titre=trim($this->titre);
		$this->langs=trim($this->langs);
		$this->level=trim($this->level);
		$this->leftmenu=trim($this->leftmenu);
		$this->perms=trim($this->perms);
		$this->enabled=trim($this->enabled);
		$this->user=trim($this->user);
		if (! $this->level) $this->level=0;

		// Check parameters
		// Put here code to add control on parameters values

        // FIXME
        // Get the max rowid in llx_menu and use it as rowid in insert because postgresql
        // may use an already used value because its internal cursor does not increase when we do
        // an insert with a forced id.
        // Two solution: Disable menu handler using database when database is postgresql or update counter when
        // enabling such menus.

		// Insert request
		$sql = "INSERT INTO ".MAIN_DB_PREFIX."menu(";
		$sql.= "menu_handler,";
		$sql.= "entity,";
		$sql.= "module,";
		$sql.= "type,";
		$sql.= "mainmenu,";
		$sql.= "fk_menu,";
		$sql.= "position,";
		$sql.= "url,";
		$sql.= "target,";
		$sql.= "titre,";
		$sql.= "langs,";
		$sql.= "level,";
		$sql.= "leftmenu,";
		$sql.= "perms,";
		$sql.= "enabled,";
		$sql.= "usertype";
		$sql.= ") VALUES (";
		$sql.= " '".$this->menu_handler."',";
		$sql.= " '".$conf->entity."',";
		$sql.= " '".$this->module."',";
		$sql.= " '".$this->type."',";
		$sql.= " '".$this->mainmenu."',";
		$sql.= " '".$this->fk_menu."',";
		$sql.= " '".$this->position."',";
		$sql.= " '".$this->url."',";
		$sql.= " '".$this->target."',";
		$sql.= " '".$this->titre."',";
		$sql.= " '".$this->langs."',";
		$sql.= " '".$this->level."',";
		$sql.= " '".$this->leftmenu."',";
		$sql.= " '".$this->perms."',";
		$sql.= " '".$this->enabled."',";
		$sql.= " '".$this->user."'";
		$sql.= ")";

		dol_syslog("Menubase::create sql=".$sql, LOG_DEBUG);
		$resql=$this->db->query($sql);
		if ($resql)
		{
			$this->id = $this->db->last_insert_id(MAIN_DB_PREFIX."menu");
			dol_syslog("Menubase::create record added has rowid=".$this->id, LOG_DEBUG);

			return $this->id;
		}
		else
		{
			$this->error="Error ".$this->db->lasterror();
			dol_syslog("Menubase::create ".$this->error, LOG_ERR);
			return -1;
		}
	}

	/*
	 *      \brief      Update database
	 *      \param      user        	User that modify
	 *      \param      notrigger	    0=no, 1=yes (no update trigger)
	 *      \return     int         	<0 if KO, >0 if OK
	 */
	function update($user=0, $notrigger=0)
	{
		global $conf, $langs;

		// Clean parameters
		$this->rowid=trim($this->rowid);
		$this->menu_handler=trim($this->menu_handler);
		$this->module=trim($this->module);
		$this->type=trim($this->type);
		$this->mainmenu=trim($this->mainmenu);
		$this->fk_menu=trim($this->fk_menu);
		$this->position=trim($this->position);
		$this->url=trim($this->url);
		$this->target=trim($this->target);
		$this->titre=trim($this->titre);
		$this->langs=trim($this->langs);
		$this->level=trim($this->level);
		$this->leftmenu=trim($this->leftmenu);
		$this->perms=trim($this->perms);
		$this->enabled=trim($this->enabled);
		$this->user=trim($this->user);

		// Check parameters
		// Put here code to add control on parameters values

		// Update request
		$sql = "UPDATE ".MAIN_DB_PREFIX."menu SET";
		$sql.= " menu_handler='".addslashes($this->menu_handler)."',";
		$sql.= " module='".addslashes($this->module)."',";
		$sql.= " type='".$this->type."',";
		$sql.= " mainmenu='".addslashes($this->mainmenu)."',";
		$sql.= " fk_menu='".$this->fk_menu."',";
		$sql.= " position='".$this->position."',";
		$sql.= " url='".addslashes($this->url)."',";
		$sql.= " target='".addslashes($this->target)."',";
		$sql.= " titre='".addslashes($this->titre)."',";
		$sql.= " langs='".addslashes($this->langs)."',";
		$sql.= " level='".$this->level."',";
		$sql.= " leftmenu='".addslashes($this->leftmenu)."',";
		$sql.= " perms='".addslashes($this->perms)."',";
		$sql.= " enabled='".addslashes($this->enabled)."',";
		$sql.= " usertype='".$this->user."'";
		$sql.= " WHERE rowid=".$this->id;

		dol_syslog("Menubase::update sql=".$sql, LOG_DEBUG);
		$resql = $this->db->query($sql);
		if (! $resql)
		{
			$this->error="Error ".$this->db->lasterror();
			dol_syslog("Menubase::update ".$this->error, LOG_ERR);
			return -1;
		}

		return 1;
	}


	/*
	 *    \brief      Load object in memory from database
	 *    \param      id          id object
	 *    \param      user        User that load
	 *    \return     int         <0 if KO, >0 if OK
	 */
	function fetch($id, $user=0)
	{
		global $langs;

		$sql = "SELECT";
		$sql.= " t.rowid,";
		$sql.= " t.menu_handler,";
		$sql.= " t.entity,";
		$sql.= " t.module,";
		$sql.= " t.type,";
		$sql.= " t.mainmenu,";
		$sql.= " t.fk_menu,";
		$sql.= " t.position,";
		$sql.= " t.url,";
		$sql.= " t.target,";
		$sql.= " t.titre,";
		$sql.= " t.langs,";
		$sql.= " t.level,";
		$sql.= " t.leftmenu,";
		$sql.= " t.perms,";
		$sql.= " t.enabled,";
		$sql.= " t.usertype as user,";
		$sql.= " t.tms";
		$sql.= " FROM ".MAIN_DB_PREFIX."menu as t";
		$sql.= " WHERE t.rowid = ".$id;

		dol_syslog("Menubase::fetch sql=".$sql, LOG_DEBUG);
		$resql=$this->db->query($sql);
		if ($resql)
		{
			if ($this->db->num_rows($resql))
			{
				$obj = $this->db->fetch_object($resql);

				$this->id    = $obj->rowid;

				$this->menu_handler = $obj->menu_handler;
				$this->entity = $obj->entity;
				$this->module = $obj->module;
				$this->type = $obj->type;
				$this->mainmenu = $obj->mainmenu;
				$this->fk_menu = $obj->fk_menu;
				$this->position = $obj->position;
				$this->url = $obj->url;
				$this->target = $obj->target;
				$this->titre = $obj->titre;
				$this->langs = $obj->langs;
				$this->level = $obj->level;
				$this->leftmenu = $obj->leftmenu;
				$this->perms = $obj->perms;
				$this->enabled = str_replace("\"","'",$obj->enabled);
				$this->user = $obj->user;
				$this->tms = $this->db->jdate($obj->tms);
			}
			$this->db->free($resql);

			return 1;
		}
		else
		{
			$this->error="Error ".$this->db->lasterror();
			dol_syslog("Menubase::fetch ".$this->error, LOG_ERR);
			return -1;
		}
	}


	/*
	 *   \brief      Delete object in database
	 *	\param      user        User that delete
	 *	\return		int			<0 if KO, >0 if OK
	 */
	function delete($user)
	{
		global $conf, $langs;

		$sql = "DELETE FROM ".MAIN_DB_PREFIX."menu";
		$sql.= " WHERE rowid=".$this->id;

		dol_syslog("Menubase::delete sql=".$sql);
		$resql = $this->db->query($sql);
		if (! $resql)
		{
			$this->error="Error ".$this->db->lasterror();
			dol_syslog("Menubase::delete ".$this->error, LOG_ERR);
			return -1;
		}

		return 1;
	}


	/**
	 *		\brief		Initialise object with example values
	 *		\remarks	id must be 0 if object instance is a specimen.
	 */
	function initAsSpecimen()
	{
		$this->id=0;

		$this->menu_handler='all';
		$this->module='specimen';
		$this->type='top';
		$this->mainmenu='';
		$this->fk_menu='0';
		$this->position='';
		$this->url='http://dummy';
		$this->target='';
		$this->titre='Specimen menu';
		$this->langs='';
		$this->level='';
		$this->leftmenu='';
		$this->perms='';
		$this->enabled='';
		$this->user='';
		$this->tms='';
	}


	/**
	 * Complete this->newmenu with menu entry found in $tab
	 *
	 * @param string $tab
	 * @param string $pere
	 * @param string $rang
	 */
	function recur($tab, $pere, $rang)
	{
		global $leftmenu;	// To be exported in dol_eval function

		//print "xx".$pere;
		$leftmenu = $this->leftmenu;
		//ballayage du tableau
		for ($x = 0; $x < count($tab); $x++)
		{
			//si un element a pour pere : $pere
			if ($tab[$x][1] == $pere)
			{
				if ($tab[$x][7])
				{
					$leftmenuConstraint = true;
					if ($tab[$x][6])
					{
						$leftmenuConstraint = verifCond($tab[$x][6]);
					}

					if ($leftmenuConstraint)
					{
						//print 'name='.$tab[$x][3].' pere='.$pere." ".$tab[$x][6];

						$this->newmenu->add((! preg_match("/^(http:\/\/|https:\/\/)/i",$tab[$x][2])) ? $tab[$x][2] : $tab[$x][2], $tab[$x][3], $rang -1, $tab[$x][4], $tab[$x][5], $tab[$x][8]);
						$this->recur($tab, $tab[$x][0], $rang +1);
					}
				}
			}
		}
	}

	/**
	 * Return all values of mainmenu where leftmenu defined
	 * in pre.inc.php must be overwritten completely by dynamic menu.
	 *
	 * @return array
	 */
	function listeMainmenu()
	{
		global $conf;

		$overwritemenufor=array();

		$sql = "SELECT DISTINCT m.mainmenu";
		$sql.= " FROM " . MAIN_DB_PREFIX . "menu as m";
		$sql.= " WHERE m.menu_handler in ('".$this->menu_handler."','all')";
		$sql.= " AND m.entity = ".$conf->entity;
		$sql.= " AND m.leftmenu != '0'";	// leftmenu exists in database so pre.inc.php must be overwritten

		//print $sql;
		$res = $this->db->query($sql);
		if ($res)
		{
			while ($menu = $this->db->fetch_array($res))
			{
				if ($menu['mainmenu']) $overwritemenufor[] = $menu['mainmenu'];
			}
		}
		else
		{
			dol_print_error($this->db);
		}

		return $overwritemenufor;
	}

	/**
	 *		Load tabMenu array
	 * 		@param		type_user		0=Internal,1=External,2=All
	 * 		@param		mainmenu		Value for mainmenu that defined top menu
	 * 		@param		menu_handler	Name of menu_handler used (auguria, eldy...)
	 * 		@return		array			Return array with menu entries for top menu
	 */
	function menuTopCharger($type_user, $mainmenu, $menu_handler)
	{
		global $langs, $user, $conf;
		global $rights;	// To export to dol_eval function

		$tabMenu=array();

		$sql = "SELECT m.rowid, m.mainmenu, m.titre, m.url, m.langs, m.perms, m.enabled, m.target";
		$sql.= " FROM ".MAIN_DB_PREFIX."menu as m";
		$sql.= " WHERE m.type = 'top'";
		$sql.= " AND m.entity = ".$conf->entity;
		$sql.= " AND m.menu_handler in('".$menu_handler."','all')";
		if ($type_user == 0) $sql.= " AND m.usertype in (0,2)";
		if ($type_user == 1) $sql.= " AND m.usertype in (1,2)";
		// If type_user == 2, no test requires
		$sql.= " ORDER BY m.position";

		dol_syslog("Menubase::menuTopCharger sql=".$sql);
		$resql = $this->db->query($sql);
		if ($resql)
		{
			$numa = $this->db->num_rows($resql);

			$a = 0;
			$b = 0;
			$oldrowid=0;
			while ($a < $numa)
			{
				$objm = $this->db->fetch_object($resql);

				// Define $chaine
				$chaine="";
				$title=$langs->trans($objm->titre);
				if ($title == $objm->titre)	// Translation not found
				{
				    if (! empty($objm->langs)) // If there is a dedicated translation file
					{
						$langs->load($objm->langs);
					}

					if (preg_match("/\//",$objm->titre)) // To manage translation when title is string1/string2
					{
						$tab_titre = explode("/",$objm->titre);
						$chaine = $langs->trans($tab_titre[0])."/".$langs->trans($tab_titre[1]);
					}
					else
					{
						$chaine = $langs->trans($objm->titre);
					}
				}
				else
				{
					$chaine=$title;
				}
				//print "x".$objm->titre."-".$chaine;

				// Define $right
				$perms = true;
				if ($objm->perms)
				{
					$perms = verifCond($objm->perms);
				}

				// Define $enabled
				$enabled = true;
				if ($objm->enabled)
				{
					$enabled = verifCond($objm->enabled);
				}

				if ($objm->rowid != $oldrowid && $oldrowid) $b++;	// Break on new entry
				$oldrowid=$objm->rowid;

				$tabMenu[$b]['rowid'] = $objm->rowid;
				$tabMenu[$b]['mainmenu'] = $objm->mainmenu;
				$tabMenu[$b]['titre'] = $chaine;	// Title
				$tabMenu[$b]['url'] = $objm->url;
				$tabMenu[$b]['atarget'] = $objm->target;
				if (! isset($tabMenu[$b]['right'])) $tabMenu[$b]['right'] = $perms;
				else $tabMenu[$b]['right'] = ($tabMenu[$b]['right'] && $perms);
				if (! isset($tabMenu[$b]['enabled'])) $tabMenu[$b]['enabled'] = $enabled;
				else $tabMenu[$b]['enabled'] = ($tabMenu[$b]['enabled'] && $enabled);
                $tabMenu[$b]['lang'] = $objm->langs;

				//var_dump($tabMenu);
				$a++;
			}
            $this->db->free($resql);
		}
		else
		{
			dol_print_error($this->db);
		}

		return $tabMenu;
	}

    /**
     * Load entries found in database in a menu array
     * @param  $newmenu        Menu array to complete
     * @param  $mainmenu       Value for mainmenu that defined top menu
     * @param  $myleftmenu     Value for left that defined leftmenu
     * @param  $type_user      0=Internal,1=External,2=All
     * @param  $menu_handler   Name of menu_handler used (auguria, eldy...)
     * @return array           Menu array completed
     */
    function menuLeftCharger($newmenu, $mainmenu, $myleftmenu, $type_user, $menu_handler)
    {
        global $langs, $user, $conf, $leftmenu; // To export to dol_eval function
        global $rights; // To export to dol_eval function

        $leftmenu=$myleftmenu;

        $this->newmenu = $newmenu;
        $this->leftmenu = $leftmenu;

        $tabMenu = array ();

        $sql = "SELECT m.rowid, m.fk_menu, m.url, m.titre, m.langs, m.perms, m.enabled, m.target, m.mainmenu, m.leftmenu";
        $sql.= " FROM ".MAIN_DB_PREFIX."menu as m";
        $sql.= " WHERE m.menu_handler in('".$menu_handler."','all')";
        $sql.= " AND m.entity = ".$conf->entity;
        if ($type_user == 0) $sql.= " AND m.usertype in (0,2)";
        if ($type_user == 1) $sql.= " AND m.usertype in (1,2)";
        // If type_user == 2, no test required
        $sql.= " ORDER BY m.position, m.rowid";

        dol_syslog("Menubase::menuLeftCharger sql=".$sql);
        $resql = $this->db->query($sql);
        if ($resql)
        {
            $numa = $this->db->num_rows($resql);

            $a = 0;
            $b = 0;
            $oldrowid=0;
            while ($a < $numa)
            {
                //$objm = $this->db->fetch_object($resql);
                $menu = $this->db->fetch_array($resql);

                // Define $chaine
                $chaine="";
                $title = $langs->trans($menu['titre']);
                if ($title == $menu['titre'])   // Translation not found
                {
                    if (! empty($menu['langs']))    // If there is a dedicated translation file
                    {
                        $langs->load($menu['langs']);
                    }

                    if (preg_match("/\//",$menu['titre'])) // To manage translation when title is string1/string2
	                {
	                    $tab_titre = explode("/",$menu['titre']);
	                    $chaine = $langs->trans($tab_titre[0])."/".$langs->trans($tab_titre[1]);
	                }
                    else
                    {
                        $chaine = $langs->trans($menu['titre']);
                    }
                }
                else
                {
                    $chaine = $title;
                }

                // Define $right
                $perms = true;
                if ($menu['perms'])
                {
                    $perms = verifCond($menu['perms']);
                    //print "verifCond rowid=".$menu['rowid']." ".$menu['right'].":".$perms."<br>\n";
                }

                // Define $enabled
                $enabled = true;
                if ($menu['enabled'])
                {
                	$enabled = verifCond($menu['enabled']);
                	if ($conf->use_javascript_ajax && $conf->global->MAIN_MENU_USE_JQUERY_ACCORDION && preg_match('/^\$leftmenu/',$menu['enabled'])) $enabled=1;
                	//print "verifCond chaine=".$chaine." rowid=".$menu['rowid']." ".$menu['enabled'].":".$enabled."<br>\n";
                }

                if ($menu['rowid'] != $oldrowid && $oldrowid) $b++; // Break on new entry
                $oldrowid=$menu['rowid'];

                $tabMenu[$b][0] = $menu['rowid'];
                $tabMenu[$b][1] = $menu['fk_menu'];
                $tabMenu[$b][2] = $menu['url'];
                if (! preg_match("/^(http:\/\/|https:\/\/)/i",$tabMenu[$b][2]))
                {
                    if (preg_match('/\?/',$tabMenu[$b][2])) $tabMenu[$b][2].='&amp;idmenu='.$menu['rowid'];
                    else $tabMenu[$b][2].='?idmenu='.$menu['rowid'];
                }
                $tabMenu[$b][3] = $chaine;
                $tabMenu[$b][5] = $menu['target'];
                $tabMenu[$b][6] = $menu['leftmenu'];
                if (! isset($tabMenu[$b][4])) $tabMenu[$b][4] = $perms;
                else $tabMenu[$b][4] = ($tabMenu[$b][4] && $perms);
                if (! isset($tabMenu[$b][7])) $tabMenu[$b][7] = $enabled;
                else $tabMenu[$b][7] = ($tabMenu[$b][7] && $enabled);
                $tabMenu[$b][8] = $menu['mainmenu'];

                $a++;
            }
            $this->db->free($resql);
        }
        else
        {
            dol_print_error($this->db);
        }

        // Get menutopid
        $menutopid='';

        $sql = "SELECT m.rowid, m.titre, m.type";
        $sql.= " FROM " . MAIN_DB_PREFIX . "menu as m";
        $sql.= " WHERE m.mainmenu = '".$mainmenu."'";
        $sql.= " AND m.menu_handler in('".$menu_handler."','all')";
        $sql.= " AND m.entity = ".$conf->entity;
        $sql.= " AND type = 'top'";

        // It should have only one response
        $resql = $this->db->query($sql);
        $menutop = $this->db->fetch_object($resql);
        if ($menutop) $menutopid=$menutop->rowid;
        $this->db->free($resql);
        //print "menutopid=".$menutopid." sql=".$sql;

        // Now edit this->newmenu to add entries in tabMenu that are in childs
        $this->recur($tabMenu, $menutopid, 1);

        return $this->newmenu;
    }

}

?>
