<?php
// Copyright (C) 2014 TeemIp
//
//   This file is part of TeemIp.
//
//   TeemIp is free software; you can redistribute it and/or modify	
//   it under the terms of the GNU Affero General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
//   TeemIp is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU Affero General Public License for more details.
//
//   You should have received a copy of the GNU Affero General Public License
//   along with TeemIp. If not, see <http://www.gnu.org/licenses/>

/**
 * @copyright   Copyright (C) 2014 TeemIp
 * @license     http://opensource.org/licenses/AGPL-3.0
 */
 
require_once('../approot.inc.php');
require_once(APPROOT.'/application/application.inc.php');
require_once(APPROOT.'/application/nicewebpage.class.inc.php');
require_once(APPROOT.'/application/wizardhelper.class.inc.php');

/**
 * Helper to determine the supported types of tickets
 */
function GetTicketClasses()
{
	$aClasses = array();
	$sPortalTicketClasses = MetaModel::GetModuleSetting('teemip-request-mgmt', 'portal_tickets', null);
	foreach (explode(',', $sPortalTicketClasses) as $sRawClass)
	{
		$sRawClass = trim($sRawClass);
		if (!MetaModel::IsValidClass($sRawClass))
		{
			throw new Exception("Class '$sRawClass' is not a valid class, please review your configuration (portal_tickets)");
		}
		if (!MetaModel::IsParentClass('Ticket', $sRawClass))
		{
			throw new Exception("Class '$sRawClass' does not inherit from Ticket, please review your configuration (portal_tickets)");
		}
		$aClasses[] = $sRawClass;
	}
	return $aClasses;
} 

/**
 * Displays the portal main menu bar
 */
 
function DisplayMainMenu(WebPage $oP)
{
	$oP->AddMenuButton('showongoing', 'Portal:ShowOngoing', '../portal/index.php?operation=show_ongoing');
	$oP->AddMenuButton('create', 'Portal:CreateNewRequest', '../portal/index.php?operation=create_request');
	$oP->AddMenuButton('showclosed', 'Portal:ShowClosed', '../portal/index.php?operation=show_closed');
	
	if (UserRights::CanChangePassword())
	{
		$oP->AddMenuButton('change_pwd', 'Portal:ChangeMyPassword', '../portal/index.php?loginop=change_pwd');
	}
}

/**
 * Displays the list of ongoing (non closed) tickets
 */

function ShowOngoingTickets(WebPage $oP)
{
	$oP->add("<div id=\"open_requests\">\n");
	$oP->add("<h1 id=\"title_open_requests\">".Dict::S('Portal:OpenRequests')."</h1>\n");
	ListOpenRequests($oP);
	$oP->add("</div>\n");

	$oP->add("<div id=\"#div_resolved_requests\">\n");
	$oP->add("<h1 id=\"#resolved_requests\">".Dict::S('Portal:ResolvedRequests')."</h1>\n");
	ListResolvedRequests($oP);
	$oP->add("</div>\n");
}

/**
 * Displays the closed tickets
 */

function ShowClosedTickets(WebPage $oP)
{
	$oP->add("<div id=\"#closed_tickets\">\n");
	ListClosedTickets($oP);
	$oP->add("</div>\n");
}

/**
 * Lists all the currently opened IP Requests for the current user
 */
 
function ListOpenRequests(WebPage $oP)
{
	$oUserOrg = GetUserOrg();
												  
	$sOQL = 'SELECT IPRequest WHERE org_id = :org_id AND status NOT IN ("resolved", "closed")';
	$oSearch = DBObjectSearch::FromOQL($sOQL);
	$iUser = UserRights::GetContactId();
	if ($iUser > 0)
	{
		$oSearch->AddCondition('caller_id', $iUser);
	}
	$oSet = new CMDBObjectSet($oSearch, array(), array('org_id' => $oUserOrg->GetKey()));
	$aZList =  explode(',', PORTAL_IPREQUEST_LIST_ZLIST);
	$oP->DisplaySet($oSet, $aZList, Dict::S('Portal:NoOpenRequest'));
}

/**
 * Lists all the currently Resolved (not "Closed") IPRequests for the current user
 */

function ListResolvedRequests(WebPage $oP)
{
	$oUserOrg = GetUserOrg();
	
	$sOQL = 'SELECT IPRequest WHERE org_id = :org_id AND status = "resolved"';
	$oSearch = DBObjectSearch::FromOQL($sOQL);
	$iUser = UserRights::GetContactId();
	if ($iUser > 0)
	{
		$oSearch->AddCondition('caller_id', $iUser);
	}
	$oSet = new CMDBObjectSet($oSearch, array(), array('org_id' => $oUserOrg->GetKey()));
	$aZList =  explode(',', PORTAL_IPREQUEST_LIST_ZLIST);
	$oP->DisplaySet($oSet, $aZList, Dict::S('Portal:NoOpenRequest'));
}

/**
 * Lists all the currently closed IPRequests
 */

function ListClosedTickets(WebPage $oP)
{
	$aAttSpecs = explode(',', PORTAL_IPREQUEST_SEARCH_CRITERIA);
	$aZList =  explode(',', PORTAL_IPREQUEST_LIST_ZLIST);
	
	$oP->DisplaySearchForm('IPRequest', $aAttSpecs, array('operation' => 'show_closed'), 'search_', false /* => not closed */);
	
	$oUserOrg = GetUserOrg();
	
	// UserRequest
	$oSearch = $oP->PostedParamsToFilter('IPRequest', $aAttSpecs, 'search_');
	if(is_null($oSearch))
	{
		$oSearch = new DBObjectSearch('IPRequest');
	}
	$oSearch->AddCondition('org_id', $oUserOrg->GetKey());
	$oSearch->AddCondition('status', 'closed');
	$iUser = UserRights::GetContactId();
	if ($iUser > 0)
	{
		$oSearch->AddCondition('caller_id', $iUser);
	}
	$oSet1 = new CMDBObjectSet($oSearch);
	$oP->add("<p>\n");
	$oP->add("<h1>".Dict::S('Portal:ClosedRequests')."</h1>\n");
	$oP->DisplaySet($oSet1, $aZList, Dict::S('Portal:NoClosedRequest'));
	$oP->add("</p>\n");
}

/**
 * Prompts the user for creating a new request
 */

function CreateRequest(WebPage $oP, Organization $oUserOrg, $oAppContext)
{
	switch($oP->GetWizardStep())
	{
		case 0:
		default:
			SelectRequestClass ($oP, $oUserOrg, $oAppContext);
		break;
		
		case 1:
			RequestCreationForm($oP, $oUserOrg);
		break;
		
		case 2:
			DoCreateRequest($oP, $oUserOrg);
		break;
	}
}

/**
 * Displays the form to select a class of IPRequest
 */

function SelectRequestClass(WebPage $oP, Organization $oUserOrg, $oAppContext)
{
	// Select IP Requests
	$sClass = 'IPRequest';
	$aSubClasses = MetaModel::EnumChildClasses($sClass, ENUM_CHILD_CLASSES_ALL); // Including the specified class itself
	$aPossibleClasses = array();
	$sRealClass = '';
	foreach($aSubClasses as $sCandidateClass)
	{
		if (!MetaModel::IsAbstract($sCandidateClass) && (UserRights::IsActionAllowed($sCandidateClass, UR_ACTION_MODIFY) == UR_ALLOWED_YES))
		{
			$aPossibleClasses[$sCandidateClass] = MetaModel::GetName($sCandidateClass);
		}
	}
	$sClassLabel = MetaModel::GetName($sClass);
	
	// Display list
	$oP->add("<div class=\"wizContainer\" id=\"form_select_ip_request\">\n");
	$oP->WizardFormStart('request_wizard', 1);
	
	$oP->add("<h1>".MetaModel::GetClassIcon($sClass)."&nbsp;".Dict::Format('UI:CreationTitle_Class', $sClassLabel)."</h1>\n");
	$oP->add('<p>'.Dict::Format('UI:SelectTheTypeOf_Class_ToCreate', $sClassLabel));
	$aDefaults = utils::ReadParam('default', array(), false, 'raw_data');
	$oP->add($oAppContext->GetForForm());
	$oP->add("<input type=\"hidden\" name=\"operation\" value=\"create_request\">");
	foreach($aDefaults as $key => $value)
	{
		if (is_array($value))
		{
			foreach($value as $key2 => $value2)
			{
				if (is_array($value2))
				{
					foreach($value2 as $key3 => $value3)
					{
						$oP->add("<input type=\"hidden\" name=\"default[$key][$key2][$key3]\" value=\"$value3\">\n");	
					}
				}
				else
				{
					$oP->add("<input type=\"hidden\" name=\"default[$key][$key2]\" value=\"$value2\">\n");	
				}
			}
		}
		else
		{
			$oP->add("<input type=\"hidden\" name=\"default[$key]\" value=\"$value\">\n");	
		}
	}
	$oP->add('<select name="class">');
	asort($aPossibleClasses);
	foreach($aPossibleClasses as $sClassName => $sClassLabel)
	{
		$sSelected = ($sClassName == $sClass) ? 'selected' : '';
		$oP->add("<option $sSelected value=\"$sClassName\">$sClassLabel</option>");
	}
	$oP->add('</select>');
	$oP->WizardFormButtons(BUTTON_BACK | BUTTON_NEXT | BUTTON_CANCEL);
	$oP->WizardFormEnd();
	$oP->add("</div>\n");
}

/**
 * Displays the form for the IPRequest creation
 */

function RequestCreationForm($oP, $oUserOrg)
{
	$sClass = utils::ReadParam('class', null, false, 'raw_data');
	if (!MetaModel::IsValidClass($sClass))
	{
		$oP->add("Warning ! Wrong class passed as parameter");
		return;
	}
	$sList = constant('PORTAL_'.strtoupper($sClass).'_LIST_ZLIST');
	$aList =  explode(',', $sList);
	$aParameters = $oP->ReadAllParams($sList);
	
	$sDescription = '';
	if (isset($aParameters['template_id']) && ($aParameters['template_id'] != 0))
	{
		$aTemplateFields = array();
		$oTemplate = MetaModel::GetObject('Template', $aParameters['template_id'], false);
		if (is_object($oTemplate))
		{
			$oFieldSearch = DBObjectSearch::FromOQL('SELECT TemplateField WHERE template_id = :template_id');
			$oFieldSearch->AllowAllData();
			$oFieldSet = new DBObjectSet($oFieldSearch, array('order' => true), array('template_id' => $oTemplate->GetKey()));
			while($oField = $oFieldSet->Fetch())
			{
				$sAttCode = $oField->Get('code');
				if (isset($aParameters[$sAttCode]))
				{
					$oField->Set('initial_value', $aParameters[$sAttCode]);
				}
				$aTemplateFields[$sAttCode] = $oField;
			}
		}
	}

	$oRequest = MetaModel::NewObject($sClass);
	$oRequest->Set('org_id', $oUserOrg->GetKey());
	$oRequest->Set('caller_id', UserRights::GetContactId());

	$iFlags = 0;
	foreach($aList as $sAttCode)
	{
		$value = '';
		if (isset($aParameters[$sAttCode]))
		{
			$value = $aParameters[$sAttCode];
			$oRequest->Set($sAttCode, $value);
		}
	}
	$aFieldsMap = array();
	foreach($aList as $sAttCode)
	{
		$value = '';
		$oAttDef = MetaModel::GetAttributeDef($sClass, $sAttCode);
		$iFlags = $oRequest->GetAttributeFlags($sAttCode);
		if (isset($aParameters[$sAttCode]))
		{
			$value = $aParameters[$sAttCode];
		}
		$aArgs = array('this' => $oRequest);
			
		$sInputId = 'attr_'.$sAttCode;
		$aFieldsMap[$sAttCode] = $sInputId;
		$sValue = "<span id=\"field_{$sInputId}\">".$oRequest->GetFormElementForField($oP, $sClass, $sAttCode, $oAttDef, $value, '', 'attr_'.$sAttCode, '', $iFlags, $aArgs).'</span>';
		$aDetails[] = array('label' => $oAttDef->GetLabel(), 'value' => $sValue);
	}

	if (!empty($aTemplateFields))
	{
		foreach ($aTemplateFields as $sAttCode =>  $oField)
		{
			if (!in_array($sAttCode, $aList))
			{
				$sValue = $oField->GetFormElement($oP, $sClass);
				if ($oField->Get('input_type') == 'hidden')
				{
					$aHidden[] = $sValue;
				}
				else
				{
					$aDetails[] = array('label' => $oField->GetAsHTML('label'), 'value' => $sValue);
				}
			}
		}
	}

	$sJsonClass = json_encode($sClass);
	$oP->add_script(
<<<EOF
// Create the object once at the beginning of the page...
	var oWizardHelper = new WizardHelper($sJsonClass, '');
EOF
);

	$oP->add_linked_script("../js/json.js");
	$oP->add_linked_script("../js/forms-json-utils.js");
	$oP->add_linked_script("../js/wizardhelper.js");
	$oP->add_linked_script("../js/wizard.utils.js");
	$oP->add_linked_script("../js/linkswidget.js");
	$oP->add_linked_script("../js/extkeywidget.js");
	$oP->add_linked_script("../js/jquery.blockUI.js");
	$oP->add("<div class=\"wizContainer\" id=\"form_request_description\">\n");
	$oP->add("<h1 id=\"title_request_form\">".Dict::S('Portal:DescriptionOfTheRequest')." ".Dict::S('UI:IPManagement:Action:Portal:'.$sClass)."</h1>\n");
	$oP->WizardFormStart('request_form', 2);

	$oP->details($aDetails);

	// Add hidden fields for known values, enabling dependant attributes to be computed correctly
	//
	foreach($oRequest->ListChanges() as $sAttCode => $value)
	{
		if (!in_array($sAttCode, $aList))
		{
			$oAttDef = MetaModel::GetAttributeDef($sClass, $sAttCode);
			if ($oAttDef->IsScalar() && $oAttDef->IsWritable())
			{
				$sValue = htmlentities($oRequest->Get($sAttCode), ENT_QUOTES, 'UTF-8');
				$oP->add("<input type=\"hidden\" id=\"attr_$sAttCode\" name=\"attr_$sAttCode\" value=\"$sValue\">");
				$aFieldsMap[$sAttCode] = 'attr_'.$sAttCode;
			}
		}
	}

	$oAttPlugin = new AttachmentPlugIn();
	$oAttPlugin->OnDisplayRelations($oRequest, $oP, true /* edit */);

	$oP->add("<input type=\"hidden\" name=\"class\" value=\"$sClass\">");
	$oP->DumpHiddenParams($aParameters, $aList);
	$oP->add("<input type=\"hidden\" name=\"operation\" value=\"create_request\">");
	$oP->WizardFormButtons(BUTTON_BACK | BUTTON_FINISH | BUTTON_CANCEL); //Back button automatically discarded if on the first page
	$oP->WizardFormEnd();
	$oP->add("</div>\n");
	$iFieldsCount = count($aFieldsMap);
	$sJsonFieldsMap = json_encode($aFieldsMap);

	$oP->add_ready_script(
<<<EOF
	oWizardHelper.SetFieldsMap($sJsonFieldsMap);
	oWizardHelper.SetFieldsCount($iFieldsCount);
    
	// Starts the validation when the page is ready
	CheckFields('request_form', false);
	$('#request_form').submit( function() {
		return OnSubmit('request_form');
	});
EOF
);
}

/**
 * Validate the parameters and create the IPRequest object (based on the page's POSTed parameters)
 */

function DoCreateRequest($oP, $oUserOrg)
{
	$sClass = utils::ReadParam('class', null, false, 'raw_data');
	if (!MetaModel::IsValidClass($sClass))
	{
		$oP->add("Warning ! Wrong class passed as parameter");
		return;
	}
	$sList = constant('PORTAL_'.strtoupper($sClass).'_LIST_ZLIST');
	$aParameters = $oP->ReadAllParams($sList);
	$sTransactionId = utils::ReadPostedParam('transaction_id', '');
	if (!utils::IsTransactionValid($sTransactionId))
	{
		$oP->add("<h1>".Dict::S('UI:Error:ObjectAlreadyCreated')."</h1>\n");
		//ShowOngoingTickets($oP);
		return;
	}
	
	$oRequest = MetaModel::NewObject($sClass);
	$oRequest->Set('org_id', $oUserOrg->GetKey());
	$oRequest->Set('caller_id', UserRights::GetContactId());
	$oRequest->UpdateObjectFromPostedForm();
	if (isset($aParameters['moreinfo']))
	{
		// There is a template, insert it into the description
		$sLogAttCode = constant('PORTAL_IPREQUEST_PUBLIC_LOG');
		$oRequest->Set($sLogAttCode, $aParameters['moreinfo']);
	}
	
	$oAttPlugin = new AttachmentPlugIn();
	$oAttPlugin->OnFormSubmit($oRequest);
	
	list($bRes, $aIssues) = $oRequest->CheckToWrite();
	if ($bRes)
	{
		$oRequest->DBInsert();
		$oP->add("<h1>".Dict::Format('UI:Title:Object_Of_Class_Created', $oRequest->GetName(), MetaModel::GetName(get_class($oRequest)))."</h1>\n");
		
		//DisplayObject($oP, $oRequest, $oUserOrg);
		ShowOngoingTickets($oP);
	}
	else
	{
		RequestCreationForm($oP, $oUserOrg);
		$sIssueDesc = Dict::Format('UI:ObjectCouldNotBeWritten', implode(', ', $aIssues));
		$oP->add_ready_script("alert('".addslashes($sIssueDesc)."');");
	}	
}

/**
 * Display object and filter on object type
 */
 
function DisplayObject($oP, $oObj, $oUserOrg)
{
	if (in_array(get_class($oObj), GetTicketClasses()))
	{
		ShowDetailsRequest($oP, $oObj);
	}
	else
	{
		throw new Exception("The class ".get_class($oObj)." is not handled through the portal");
	}
}

/**
 * Displays the details of a request
 */

function ShowDetailsRequest(WebPage $oP, $oObj)
{
	$sClass = get_class($oObj);
	$sLogAttCode = constant('PORTAL_IPREQUEST_PUBLIC_LOG');
	$sUserCommentAttCode = constant('PORTAL_IPREQUEST_USER_COMMENT');
	
	$bIsReopenButton = false;
	$bIsCloseButton = false;
	$bEditAttachments = false;
	$aEditAtt = array(); // List of attributes editable in the main form
	switch($oObj->GetState())
	{
		case 'new':
		case 'rejected':
		case 'assigned':
			$aEditAtt = array(
				$sLogAttCode => '????'
			);
			$bEditAttachments = true;
		break;
		
		case 'resolved':
			$aEditAtt = array();
			if (array_key_exists('ev_reopen', MetaModel::EnumStimuli($sClass)))
			{
				$bIsReopenButton = true;
				MakeStimulusForm($oP, $oObj, 'ev_reopen', array($sLogAttCode));
			}
			$bIsCloseButton = true;
			MakeStimulusForm($oP, $oObj, 'ev_close', array($sUserCommentAttCode));
		break;
		
		case 'closed':
		default:
		break;
	}
	
// REFACTORISER LA MISE EN FORME
	$oP->add("<h1 id=\"title_request_details\">".$oObj->GetIcon()."&nbsp;".Dict::Format('Portal:TitleRequestDetailsFor_Request', $oObj->GetName())." ".Dict::S('UI:IPManagement:Action:Portal:'.$sClass)."</h1>\n");

	$aAttList = json_decode(constant('PORTAL_'.strtoupper($sClass).'_DETAILS_ZLIST'), true);	
	
	switch($oObj->GetState())
	{
		case 'closed':
		$aAttList['centered'][] = $sUserCommentAttCode;
	}

	// Remove the edited attribute from the shown attributes
	//
	foreach($aEditAtt as $sAttCode => $foo)
	{
		foreach($aAttList as $col => $aColumn)
		{
			if (in_array($sAttCode, $aColumn))
			{
				if(($index = array_search($sAttCode, $aColumn)) !== false)
				{
					unset($aAttList[$col][$index]);
				}
			}
		}
	}
	
	$oP->add("<div class=\"wizContainer\" id=\"form_commment_request\">\n");
	$oP->WizardFormStart('request_form', null);
	
	$oP->add('<div id="request_details">');
	$oP->add('<table id="request_details_table">');
	
	$oP->add('<tr>');
	$oP->add('<td style="vertical-align:top;">');
	$oP->DisplayObjectDetails($oObj, $aAttList['col:left']);
	$oP->add('</td>');
	$oP->add('<td style="vertical-align:top;">');
	$oP->DisplayObjectDetails($oObj, $aAttList['col:right']);
	$oP->add('</td>');
	$oP->add('</tr>');
	if (array_key_exists('centered', $aAttList))
	{
		$oP->add('<tr>');
		$oP->add('<td style="vertical-align:top;" colspan="2">');
		$oP->DisplayObjectDetails($oObj, $aAttList['centered']);
		$oP->add('</td>');
		$oP->add('</tr>');
	}
	
// REFACTORISER
	$oP->add('<tr>');
	$oP->add('<td colspan="2" style="vertical-align:top;">');
	$oAttPlugin = new AttachmentPlugIn();
	if ($bEditAttachments)
	{
		$oAttPlugin->EnableDelete(false);
		$oAttPlugin->OnDisplayRelations($oObj, $oP, true /* edit */);
	}
	else
	{
		$oAttPlugin->OnDisplayRelations($oObj, $oP, false /* read */);
	}
	$oP->add('</td>');
	$oP->add('</tr>');
	
	$oP->add('<tr>');
	$oP->add('<td colspan="2" style="vertical-align:top;">');
	
	//$oP->add("<form action=\"../portal/index.php\" id=\"request_form\" method=\"post\">\n");
	//$oP->add('<table id=""><tr><td style="vertical-align:top;">');
	//$oP->add("<h1 id=\"title_request_details\">".Dict::Format('Portal:CommentsFor_Request', $oObj->GetName())."</h1>\n");
	$oP->add("<input type=\"hidden\" name=\"class\" value=\"$sClass\">");
	$oP->add("<input type=\"hidden\" name=\"id\" value=\"".$oObj->GetKey()."\">");
	$oP->add("<input type=\"hidden\" name=\"operation\" value=\"update_request\">");
	$oP->add("<input type=\"hidden\" id=\"stimulus_to_apply\" name=\"apply_stimulus\" value=\"\">\n");
	$oP->add_script(
<<<EOF
	function SetStimulusToApply(sStimulusCode)
	{
		$('#stimulus_to_apply').val(sStimulusCode);
	}
EOF
);
	$aEditFields = array(); // Intermediate array to avoid code duplication while splitting btw ticket_log and the rest
	foreach($aEditAtt as $sAttCode => $foo)
	{
		$sValue = $oObj->Get($sAttCode);
		$sDisplayValue = $oObj->GetEditValue($sAttCode);
		$aArgs = array('this' => $oObj, 'formPrefix' => '');
		$oAttDef = MetaModel::GetAttributeDef($sClass, $sAttCode);
		$sInputId = 'input_'.$sAttCode;
		$sHTMLValue = "<span id=\"field_{$sInputId}\">".cmdbAbstractObject::GetFormElementForField($oP, $sClass, $sAttCode, $oAttDef, $sValue, $sDisplayValue, $sInputId, '', 0 /*$iFlags*/, $aArgs).'</span>';
		
		$aEditFields[$sAttCode] = array(
			'label' => MetaModel::GetLabel($sClass, $sAttCode),
			'value' => $sHTMLValue
		);
	}
	foreach($aEditFields as $sAttCode => $aFieldSpec)
	{
		if ($sAttCode == $sLogAttCode)
		{
			// Skip, the public log will be displayed below the buttons
			continue;
		}
		$oP->add("<div class=\"edit_item\">");
		$oP->add('<h1>'.$aFieldSpec['label'].'</h1>');
		$oP->add($aFieldSpec['value']);
		$oP->add('</div>');
	}
	if($bIsReopenButton)
	{
		$sStimulusCode = 'ev_reopen';
		$sTitle = addslashes(Dict::S('Portal:Button:ReopenTicket'));
		$sOk = addslashes(Dict::S('UI:Button:Ok'));
		$oP->p('<input type="button" onClick="RunStimulusDialog(\''.$sStimulusCode.'\', \''.$sTitle.'\', \''.$sOk.'\');" value="'.$sTitle.'...">');
	}
	if($bIsCloseButton)
	{
		$sStimulusCode = 'ev_close';
		$sTitle = addslashes(Dict::S('Portal:Button:CloseTicket'));
		$sOk = addslashes(Dict::S('UI:Button:Ok'));
		$oP->p('<input type="button" onClick="RunStimulusDialog(\''.$sStimulusCode.'\', \''.$sTitle.'\', \''.$sOk.'\');" value="'.$sTitle.'...">');
	}
	elseif (count($aEditAtt) > 0)
	{
		$oP->p('<input type="submit" value="'.Dict::S('Portal:Button:UpdateRequest').'">');
	}
		
	$oP->add('</td>');
	$oP->add('</tr>');
	
	$oP->add('<tr>');
	$oP->add('<td colspan="2" style="vertical-align:top;">');
	if (isset($aEditFields[$sLogAttCode]))
	{
		$oP->add("<div class=\"edit_item\">");
		$oP->add('<h1>'.$aEditFields[$sLogAttCode]['label'].'</h1>');
		$oP->add($aEditFields[$sLogAttCode]['value']);
		$oP->add('</div>');
	}
	else
	{
		$oP->add('<h1>'.MetaModel::GetLabel($sClass, $sLogAttCode).'</h1>');
		$oP->add($oObj->GetAsHTML($sLogAttCode));
	}
	$oP->add('</td>');
	$oP->add('</tr>');

	$oP->add('</table>');
	$oP->add('</div>');

	$oP->WizardFormEnd();
	$oP->add('</div>');
}	

/**
 * Create form to apply a stimulus
 */
function MakeStimulusForm(WebPage $oP, $oObj, $sStimulusCode, $aEditAtt)
{
	static $bHasStimulusForm = false;

	$sDialogId = $sStimulusCode."_dialog";
	$sFormId = $sStimulusCode."_form";
	$sCancelButtonLabel = Dict::S('UI:Button:Cancel');

	$oP->add('<div id="'.$sDialogId.'" style="display: none;">');
	$sClass = get_class($oObj);

	$oP->add('<form id="'.$sFormId.'" method="post">');
	$sTransactionId = utils::GetNewTransactionId();
	$oP->add("<input type=\"hidden\" id=\"transaction_id\" name=\"transaction_id\" value=\"$sTransactionId\">\n");
	$oP->add("<input type=\"hidden\" name=\"class\" value=\"$sClass\">");
	$oP->add("<input type=\"hidden\" name=\"id\" value=\"".$oObj->GetKey()."\">");
	$oP->add("<input type=\"hidden\" name=\"operation\" value=\"update_request\">");
	$oP->add("<input type=\"hidden\" id=\"stimulus_to_apply\" name=\"apply_stimulus\" value=\"$sStimulusCode\">\n");
	
	foreach($aEditAtt as $sAttCode)
	{
		$sValue = $oObj->Get($sAttCode);
		$sDisplayValue = $oObj->GetEditValue($sAttCode);
		$aArgs = array('this' => $oObj, 'formPrefix' => '');
		$oAttDef = MetaModel::GetAttributeDef($sClass, $sAttCode);
		$sInputId = 'input_'.$sAttCode;
		$sHTMLValue = "<span id=\"field_{$sStimulusCode}_{$sInputId}\">".cmdbAbstractObject::GetFormElementForField($oP, $sClass, $sAttCode, $oAttDef, $sValue, $sDisplayValue, $sInputId, '', 0 /*$iFlags*/, $aArgs).'</span>';

		$oP->add('<h1>'.MetaModel::GetLabel($sClass, $sAttCode).'</h1>');
		$oP->add($sHTMLValue);
	}
	$oP->add('</form>');
	$oP->add('</div>');

	if (!$bHasStimulusForm)
	{
		$bHasStimulusForm = true;
		$oP->add_script(
<<<EOF

function RunStimulusDialog(sStimulusCode, sTitle, sOkButtonLabel)
{
	$('#'+sStimulusCode+'_dialog').dialog({
		height: 'auto',
		width: 'auto',
		modal: true,
		title: sTitle,
		buttons: [
		{ text: sOkButtonLabel, click: function() {
			$(this).find('#'+sStimulusCode+'_form').submit();
		} },
		{ text: "$sCancelButtonLabel", click: function() {
			$(this).dialog( "close" );
		} },
		],
	});
}
EOF
		);
	}
}

/**
 * Get The organization of the current user (i.e. the organization of its contact)
 */

function GetUserOrg()
{
	$oOrg = null;
	$iContactId = UserRights::GetContactId();
	$oContact = MetaModel::GetObject('Contact', $iContactId, false); // false => Can fail
	if (is_object($oContact))
	{
		$oOrg = MetaModel::GetObject('Organization', $oContact->Get('org_id'), false); // false => can fail
	}
	else
	{
		throw new Exception(Dict::S('Portal:ErrorNoContactForThisUser'));
	}
	return $oOrg;
}

/**
 * Determine if the current user can be considered as being a portal power user
 */
  
function IsPowerUSer()
{
	$iUserID = UserRights::GetUserId();
	$sOQLprofile = "SELECT URP_Profiles AS p JOIN URP_UserProfile AS up ON up.profileid=p.id WHERE up.userid = :user AND p.name = :profile";
	$oProfileSet = new DBObjectSet(
		DBObjectSearch::FromOQL($sOQLprofile),
		array(),
		array(
			'user' => $iUserID,
			'profile' => PORTAL_POWER_USER_PROFILE,
		)
	);	
	$bRes = ($oProfileSet->count() > 0);
	return $bRes;
}

/*****************************************************************
 * 
 * Main user interface pages for TeemIp Portal starts here
 *
 * ***************************************************************/
 
try
{
	require_once(APPROOT.'/application/startup.inc.php');
	require_once(APPROOT.'/application/portalwebpage.class.inc.php');
	$oAppContext = new ApplicationContext();
	$sOperation = utils::ReadParam('operation', '');
	
	require_once(APPROOT.'/application/loginwebpage.class.inc.php');
	LoginWebPage::DoLogin(false /* bMustBeAdmin */, true /* IsAllowedToPortalUsers */); // Check user rights and prompt if needed
	
	ApplicationContext::SetUrlMakerClass('MyPortalURLMaker');
	
	$oUserOrg = GetUserOrg();
	
	$sCode = $oUserOrg->Get('code');
	$sAlternateStylesheet = '';
	if (@file_exists("./$sCode/portal.css"))
	{
		$sAlternateStylesheet = "$sCode";
	}
	
	$oP = new PortalWebPage(Dict::S('Portal:Title'), $sAlternateStylesheet);
	
	$oP->EnableDisconnectButton(utils::CanLogOff());
	$oP->SetWelcomeMessage(Dict::Format('Portal:WelcomeUserOrg', UserRights::GetUserFriendlyName(), $oUserOrg->GetName()));
	
	if (is_object($oUserOrg))
	{
		if (MetaModel::IsValidClass('IPRequest'))
		{
			DisplayMainMenu($oP);
			switch($sOperation)
			{
				case 'create_request':
					CreateRequest($oP, $oUserOrg, $oAppContext);
				break;
				
				case 'details':
					$oObj = $oP->FindObjectFromArgs(array('IPRequest'));
					DisplayObject($oP, $oObj, $oUserOrg);
				break;
				
				case 'update_request':
					$oObj = $oP->FindObjectFromArgs(array('IPRequest'));
					$sClass = get_class($oObj);
					$sPortalTicketClasses = MetaModel::GetModuleSetting('teemip-request-mgmt', 'portal_tickets', null);
					if (strpos($sPortalTicketClasses, $sClass) !== false)
					{
						$aAttList = array(
							constant('PORTAL_IPREQUEST_PUBLIC_LOG'),
							constant('PORTAL_IPREQUEST_USER_COMMENT')
						);
					}
					else
					{
						throw new Exception("Implementation issue: unexpected class '".$sClass."'");
					}
					try
					{
						$oP->DoUpdateObjectFromPostedForm($oObj, $aAttList);
					}
					catch(TransactionException $e)
					{
						$oP->add("<h1>".Dict::S('UI:Error:ObjectAlreadyUpdated')."</h1>\n");
					}
					DisplayObject($oP, $oObj, $oUserOrg);
				break;
				
				case 'show_closed':
					ShowClosedTickets($oP);
				break;
				
				case 'show_ongoing':
				default:
					ShowOngoingTickets($oP);
				break;
			}
		}
		else
		{
			$oP->add("<h1>".Dict::Format('Portal:NoRequestMgmt', UserRights::GetUserFriendlyName())."</h1>\n");
		}
	}
	$oP->output();
}
catch(CoreException $e)
{
	require_once(APPROOT.'/setup/setuppage.class.inc.php');
	$oP = new SetupPage(Dict::S('UI:PageTitle:FatalError'));
	$oP->add("<h1>".Dict::S('UI:FatalErrorMessage')."</h1>\n");	
	$oP->error(Dict::Format('UI:Error_Details', $e->getHtmlDesc()));	
	$oP->output();
	
	if (MetaModel::IsLogEnabledIssue())
	{
		if (MetaModel::IsValidClass('EventIssue'))
		{
			try
			{
				$oLog = new EventIssue();
				
				$oLog->Set('message', $e->getMessage());
				$oLog->Set('userinfo', '');
				$oLog->Set('issue', $e->GetIssue());
				$oLog->Set('impact', 'Page could not be displayed');
				$oLog->Set('callstack', $e->getTrace());
				$oLog->Set('data', $e->getContextData());
				$oLog->DBInsertNoReload();
			}
			catch(Exception $e)
			{
				IssueLog::Error("Failed to log issue into the DB");
			}
		}
		
		IssueLog::Error($e->getMessage());
	}
	
	// For debugging only
	//throw $e;
}
catch(Exception $e)
{
	require_once(APPROOT.'/setup/setuppage.class.inc.php');
	$oP = new SetupPage(Dict::S('UI:PageTitle:FatalError'));
	$oP->add("<h1>".Dict::S('UI:FatalErrorMessage')."</h1>\n");	
	$oP->error(Dict::Format('UI:Error_Details', $e->getMessage()));	
	$oP->output();
	
	if (MetaModel::IsLogEnabledIssue())
	{
		if (MetaModel::IsValidClass('EventIssue'))
		{
			try
			{
				$oLog = new EventIssue();
				
				$oLog->Set('message', $e->getMessage());
				$oLog->Set('userinfo', '');
				$oLog->Set('issue', 'PHP Exception');
				$oLog->Set('impact', 'Page could not be displayed');
				$oLog->Set('callstack', $e->getTrace());
				$oLog->Set('data', array());
				$oLog->DBInsertNoReload();
			}
			catch(Exception $e)
			{
				IssueLog::Error("Failed to log issue into the DB");
			}
		}
		
		IssueLog::Error($e->getMessage());
	}
}
