<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Category;

use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\Api\Exception\JsonRpc\ParameterException;

/**
 * Class Entry
 *
 * @package idoit\Module\Api\Category
 */
class Entry
{
    /**
     * Category constant
     *
     * @var string
     */
    protected $categoryConstant;

    /**
     * @var \isys_cmdb_dao_category
     */
    protected $categoryDao;

    /**
     * Category Descriptor
     *
     * @var Descriptor
     */
    protected $categoryDescriptor;

    /**
     * Entry data
     *
     * @var array
     */
    protected $entryData;

    /**
     * EntryId
     *
     * @var int|null
     */
    protected $entryId;

    /**
     * EntryStatusId
     *
     * @var int
     */
    protected $entryStatus;

    /**
     * ObjectId
     *
     * @var int
     */
    protected $objectId;

    /**
     * @var
     */
    protected $targetStatus;

    /**
     * @return int
     */
    public function getObjectId()
    {
        return $this->objectId;
    }

    /**
     * @param int $objectId
     *
     * @return Entry
     */
    public function setObjectId($objectId)
    {
        $this->objectId = $objectId;

        return $this;
    }

    /**
     * @return string
     */
    public function getCategoryConstant()
    {
        return $this->categoryConstant;
    }

    /**
     * @param string $categoryConstant
     *
     * @return Entry
     */
    public function setCategoryConstant($categoryConstant)
    {
        $this->categoryConstant = $categoryConstant;

        return $this;
    }

    /**
     * @return int|null
     */
    public function getEntryId()
    {
        return $this->entryId;
    }

    /**
     * @param int|null $entryId
     *
     * @return Entry
     */
    public function setEntryId($entryId)
    {
        $this->entryId = $entryId;

        return $this;
    }

    /**
     * @return Descriptor
     */
    public function getCategoryDescriptor()
    {
        return $this->categoryDescriptor;
    }

    /**
     * @param Descriptor $categoryDescriptor
     *
     * @return Entry
     */
    public function setCategoryDescriptor(Descriptor $categoryDescriptor)
    {
        $this->categoryDescriptor = $categoryDescriptor;

        return $this;
    }

    /**
     * @return \isys_cmdb_dao_category
     */
    public function getCategoryDao()
    {
        return $this->categoryDao;
    }

    /**
     * @param \isys_cmdb_dao_category $categoryDao
     *
     * @return Entry
     */
    public function setCategoryDao(\isys_cmdb_dao_category $categoryDao)
    {
        $this->categoryDao = $categoryDao;

        return $this;
    }

    /**
     * @return int
     */
    public function getEntryStatus()
    {
        return $this->entryStatus;
    }

    /**
     * @param int $entryStatus
     *
     * @return Entry
     */
    public function setEntryStatus($entryStatus)
    {
        $this->entryStatus = $entryStatus;

        return $this;
    }

    /**
     * @return int
     */
    public function getTargetStatus()
    {
        return $this->targetStatus;
    }

    /**
     * @param int $targetStatus
     *
     * @return Entry
     */
    public function setTargetStatus($targetStatus)
    {
        $this->targetStatus = $targetStatus;

        return $this;
    }

    /**
     * @return array
     */
    public function getEntryData()
    {
        return $this->entryData;
    }

    /**
     * @param array $entryData
     *
     * @return Entry
     */
    public function setEntryData($entryData)
    {
        $this->entryData = $entryData;

        return $this;
    }

    /**
     * Recycle entry stepwise
     *
     * @return bool
     * @throws InternalErrorException
     * @throws ParameterException
     * @throws \isys_exception_cmdb
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     * @throws \isys_exception_general
     * @throws \Exception
     */
    public function recycle()
    {
        return $this->rank(C__RECORD_STATUS__NORMAL);
    }

    /**
     * Delete entry stepwise
     *
     * @return bool
     * @throws InternalErrorException
     * @throws ParameterException
     * @throws \isys_exception_cmdb
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     * @throws \isys_exception_general
     * @throws \Exception
     */
    public function delete()
    {
       return $this->rank(C__RECORD_STATUS__DELETED);
    }

    /**
     * Archive entry stepwise
     *
     * @return bool
     * @throws InternalErrorException
     * @throws ParameterException
     * @throws \isys_exception_cmdb
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     * @throws \isys_exception_general
     * @throws \Exception
     */
    public function archive()
    {
        return $this->rank(C__RECORD_STATUS__ARCHIVED);
    }

    /**
     * Purge entry
     *
     * @return bool
     * @throws InternalErrorException
     * @throws ParameterException
     * @throws \isys_exception_cmdb
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     * @throws \isys_exception_general
     * @throws \Exception
     */
    public function purge()
    {
        return $this->rank(C__RECORD_STATUS__PURGE);
    }

    /**
     * Entry ranker
     *
     * @param int $targetStatus
     *
     * @return bool
     * @throws \Exception
     */
    private function rank($targetStatus)
    {
        // Check whether category is multi valued
        if (!$this->categoryDao->is_multivalued() && $targetStatus !== C__RECORD_STATUS__PURGE) {
            throw new \Exception('Unable to rank single value category entry.');
        }

        // Special purge handling
        if ($targetStatus === C__RECORD_STATUS__PURGE) {
            // Try to rank entry in category
            if (!$this->categoryDao->rank_record($this->entryId, C__CMDB__RANK__DIRECTION_DELETE, $this->categoryDao->get_table(), null, true)) {
                throw new InternalErrorException('Unable to purge entry.');
            }

            // Store target status id
            $this->setTargetStatus(C__RECORD_STATUS__PURGE);

            return true;
        }

        // Check whether entry status could be detected
        if (empty($this->getEntryStatus())) {
            throw new InternalErrorException('Unable to determine entry status id.');
        }

        // Calculate status difference
        $statusDifference = $targetStatus - $this->getEntryStatus();

        // Check whether entry is already in desired status
        if ($statusDifference === 0) {
            throw new \Exception('Desired and actual status of entry are already the same.');
        }

        // Calculate direction
        $direction = $statusDifference > 0 ? C__CMDB__RANK__DIRECTION_DELETE : C__CMDB__RANK__DIRECTION_RECYCLE;

        // Iterate needed transitions
        for ($i = 0 ; $i < abs($statusDifference) ; $i++) {
            // Rank record
            if (!$this->categoryDao->rank_records([$this->getEntryId()], $direction, $this->categoryDao->get_table())) {
                throw new \Exception('Unable to transition category status.');
            }
        }

        $this->setTargetStatus($targetStatus);

        return true;
    }

    /**
     * Initialize Ranker instance
     *
     * @return Entry
     * @throws \Exception
     */
    protected function initialize()
    {
        // Validate objectId
        if (!is_int($this->objectId) || $this->objectId < 0) {
            throw new \Exception('ObjectId has to be a positive numeric value.');
        }

        // Validate category constant
        if (empty($this->categoryConstant) || !is_string($this->categoryConstant) || !defined($this->categoryConstant)) {
            throw new \Exception('Please provide an valid category constant.');
        }

        // Create category descriptor
        $categoryDescriptor = Descriptor::byConstant($this->categoryConstant);
        $categoryDaoInstance = $categoryDescriptor->getDaoInstance();

        // Check whether category is virtual
        if ($categoryDescriptor->isVirtual()) {
            throw new \Exception('Unable to handle virtual categories.');
        }

        // Set category descriptor
        $this->setCategoryDescriptor($categoryDescriptor)
            ->setCategoryDao($categoryDaoInstance);

        if ($this->categoryDao->is_multivalued() && empty($this->entryId)) {
            throw new \Exception('EntryId is required in multivalue category context. Please provide a valid entryId.');
        }

        // Get category entry data
        $entryData = $this->categoryDao->get_data($this->entryId, $this->objectId, null)
            ->get_row();

        // Check whether archived/delete categoryId was found
        if (empty($entryData)) {
            if ($this->categoryDao->is_multivalued()) {
                $message = 'Unable to find entry for id ' . $this->entryId . '. Please ensure that entry is owned by object ' . $this->objectId . ' and has a valid status.';
            } else {
                $message = 'Object ' . $this->objectId . ' does not own an entry in category \'' . $this->categoryDescriptor->getTranslatedTitle() . '\'.';
            }

            throw new ParameterException($message);
        }

        // Store information and return instance
        return $this->setEntryId($entryData[$this->categoryDao->get_table() . '__id'])
            ->setEntryStatus($entryData[$this->categoryDao->get_table() . '__status'])
            ->setEntryData($entryData, $this->categoryDao->get_table());
    }

    /**
     * Ranker constructor.
     *
     * @param int      $objectId
     * @param string   $categoryConstant
     * @param int|null $entryId
     *
     * @throws \Exception
     */
    public function __construct($objectId, $categoryConstant, $entryId)
    {
        // Store information
        $this->setObjectId($objectId)
            ->setCategoryConstant($categoryConstant)
            ->setEntryId($entryId);

        $this->initialize();
    }
}
