<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Request;

use idoit\Module\Api\Exception\JsonRpc\ParameterException;
use idoit\Module\Api\Exception\JsonRpc\RequestException;

/**
 * Class RequestStore
 *
 * @package idoit\Module\Api\Request
 */
class RequestStore
{
    /**
     * Raw request
     *
     * @var array
     */
    private $rawRequests = [];

    /**
     * Request store
     *
     * @var RpcRequest[]
     */
    private $requestStore = [];

    /**
     * Get raw request
     *
     * @return array
     */
    public function getRawRequests()
    {
        return $this->rawRequests;
    }

    /**
     * Initialize request store
     *
     * @throws \Exception
     */
    public function initialize()
    {
        // Check whether raw request is an array
        if (!is_array($this->rawRequests)) {
            throw new \Exception('Raw request is not an array.');
        }

        if (count($this->rawRequests) === 0) {
            throw new RequestException('Batch does not include any requests.');
        }

        // Get raw requests
        $rawRequests = $this->rawRequests;

        // Check whether request is batched
        if (!isset($rawRequests[0])) {
            $rawRequests = [$rawRequests];
        }

        // Store of spotted request ids
        $requestIds = [];
        $apiKeys = [];

        // Transform raw requests into RpcRequests
        foreach ($rawRequests as $rawRequest) {
            if (!is_array($rawRequest)) {
                $this->addRequest(new RpcRequest([]));
                continue;
            }

            // Check whether id was spotted already
            if (!empty($rawRequest['id']) && in_array($rawRequest['id'], $requestIds)) {
                throw new \Exception('Duplicate request id for key \'' . $rawRequest['id']. '\' detected in batch request.');
            }

            if (array_key_exists('id', $rawRequest) && !is_int($rawRequest['id']) && !is_string($rawRequest['id']) || $rawRequest['id'] === '') {
                throw new \Exception('Request contains an invalid request id. Please provide a integer or string value.');
            }

            // Mark id as already used
            $requestIds[] = $rawRequest['id'];

            if (!empty($rawRequest['params']['apikey'])) {
                $apiKeys[] = $rawRequest['params']['apikey'];
            }


            if (!is_array($rawRequest)) {
                $rawRequest = [];
            }

            // Add request
            $this->addRequest(new RpcRequest($rawRequest));
        }

        // Check whether multiple api keys were used
        if (count(array_unique($apiKeys)) > 1) {
            throw (new ParameterException('Batch requests should only reference a single tenant by using the same api keys in each request. Usage of multiple api keys is not allowed: ' . implode(', ', $apiKeys)))->setData($apiKeys);
        }
    }

    /**
     * Get RPC requests
     *
     * @return RpcRequest[]
     */
    public function getRequests()
    {
        return $this->requestStore;
    }

    /**
     * Check whether request is batched
     *
     * @return bool
     */
    public function isBatchRequest()
    {
        return count($this->requestStore) > 1;
    }

    /**
     * Check whether store has requests
     *
     * @return bool
     */
    public function hasRequests()
    {
        return !!count($this->requestStore);
    }

    /**
     * Add request to the store
     *
     * @param AbstractRequest $request
     */
    private function addRequest(AbstractRequest $request)
    {
        $this->requestStore[] = $request;
    }

    /**
     * RequestStore constructor.
     *
     * @param array $requests
     *
     * @throws \Exception
     */
    public function __construct(array $requests)
    {
        $this->rawRequests = $requests;

        $this->initialize();
    }
}
