<?

/************************************************************************/
/* FlatNuke - Flat Text Based Content Management System                 */
/* ============================================                         */
/*                                                                      */
/* Copyright (c) 2003-2005 by Simone Vellei                             */
/* http://flatnuke.sourceforge.net                                      */
/*                                                                      */
/* This program is free software. You can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License.       */
/************************************************************************/


/**
 * File delle funzioni condivise tra il forum e il core di FlatNuke
 *
 * Questo file contiene le procedure di sistema necessarie al funzionamento
 * di FlatNuke e del suo forum.
 *
 * @package Funzioni_di_sistema
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 * @version 2.5.7
 * @copyright Copyright (c) 2003-2005
 * @license http://opensource.org/licenses/gpl-license.php GNU General Public License
 */


/**
 * Restituisce la versione attuale di Flatnuke
 *
 * @since 2.5.7
 *
 * @return constant Versione di Flatnuke
 */
function get_fn_version(){
   define("FN_VERSION", "2.5.7");
   return FN_VERSION;
}


/**
 * A collection of easy to include PHP functions that sanitize user inputs
 *
 * @author The Open Web Application Security Project - {@link http://www.owasp.org/software/labs/phpfilters.html}
 * @license http://opensource.org/licenses/gpl-license.php GNU General Public License
 */
include_once("include/php_filters/sanitize.php");


/**
 * Elimina i caratteri che possono provocare problemi di sicurezza
 *
 * Elimina dalla stringa passata come parametro i caratteri che possono
 * provocare problemi di sicurezza se eseguiti in concomitanza con codice
 * maligno.
 *
 * @author Marco Segato <segatom@users.sourceforge.net>
 * @since 2.5.7
 *
 * @param string $string Stringa da verificare
 * @return string $string Stringa elaborata
 */
function fnsanitize($string) {

	$string = str_replace(chr(10), "", $string);
	$string = str_replace(chr(13), "", $string);
	$string = str_replace(chr(00), "", $string);
	$string = str_replace("%00", "", $string);
	$string = str_replace("[", "&#91;", $string);
	$string = str_replace("]", "&#93;", $string);

	return $string;
}

/**
 * Restituisce un parametro POST o GET
 *
 * Restituisce un parametro controllato con fnsanitize derivante dai
 * metodi POST o GET.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 * @since 2.5.7
 *
 * @param string $param Nome del parametro
 * @param string $opt Opzione per specificare metodo (POST,GET,ALL, COOKIE, SESSION)
 * @param pointer $sanitize Puntatore a funzione che sanitizza il parametro
 * @return string $param Contenuto del parametro
 */
function getparam($param, $opt, $sanitize) {

	if(!isset($sanitize) OR ($sanitize==SAN_FLAT))
		$sanitize="fnsanitize";
	else if($sanitize==SAN_SYST)
		$sanitize="sanitize_system_string";
	else if($sanitize==SAN_PARA)
		$sanitize="sanitize_paranoid_string";
	else if($sanitize==SAN_NULL)
		$sanitize="sanitize_null_string";
	else if($sanitize==SAN_HTML)
		$sanitize="sanitize_html_string";

	if($opt==PAR_ALL) {
		if(isset($_GET[$param]))
			return($sanitize($_GET[$param]));
		else {
			if(isset($_POST[$param]))
				return($sanitize($_POST[$param]));
			else
				return("");
		}
	}

	else if($opt==PAR_POST) {
		if(isset($_POST[$param]))
			return($sanitize($_POST[$param]));
		else
			return("");
	}

	else if($opt==PAR_GET) {
		if(isset($_GET[$param]))
			return($sanitize($_GET[$param]));
		else
			return("");
	}

	else if($opt==PAR_COOKIE) {
		if(isset($_COOKIE[$param]))
			return($sanitize($_COOKIE[$param]));
		else
			return("");
	}

	else if($opt==PAR_SESSION) {
		if(isset($_SESSION[$param]))
			return($sanitize($_SESSION[$param]));
		else
			return("");
	}

	else if($opt==PAR_SERVER) {
		if(isset($_SERVER[$param]))
			return($sanitize($_SERVER[$param]));
		else
			return("");
	}

	else if($opt==PAR_NULL) {
		if(isset($param))
			return($sanitize($param));
		else
			return("");
	}

	return("");
}


/**
 * Restituisce il codice di controllo codificato MD5 di un utente
 *
 * Restituisce il codice di controllo dell'utente passato come parametro,
 * mantenendo la codifica MD5.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $user Nome assoluto del profilo utente da elaborare
 * @return string Codice di controllo codificato MD5 dell'utente
 */
function getpass($user) {
	$user = getparam($user,PAR_NULL, SAN_FLAT);
	$fd = file("users/".$user.".php");
	$pass = str_replace("#", "", str_replace("\n", "", $fd[1]));
	return ($pass);
}


/**
 * Applica un semaforo ad un file
 *
 * Applica un semaforo al file passato come parametro, impedendo qualsiasi
 * azione fintanto che non sar spento con la chiave MD5 corretta.
 * NB: la gestione dei semafori deve essere supportata dal proprio webserver.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $user Nome assoluto del file da bloccare
 * @return string ID del semaforo
 */
function lock($filename) {
	/* uncomment this if your webserver
	// supports semaphores
	// get semaphore key
	$sem_key = md5($filename);
	// get semaphore identifier
	$sem_id = sem_get($sem_key, 1);
	// acquire semaphore lock
	sem_acquire($sem_id);
	// return sem_id
	return $sem_id;
	*/
}


/**
 * Rilascia un semaforo applicato ad un file
 *
 * Rilascia il semaforo con ID uguale a quello passato come parametro,
 * permettendo nuovamente l'accesso al relativo file.
 * NB: la gestione dei semafori deve essere supportata dal proprio webserver.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $user ID del semaforo
 */
function unlock($sem_id) {
	/* uncomment this if your webserver
	// supports semaphores
	// release semaphore lock
	sem_release($sem_id);
	*/
}


/**
 * Restituisce il codice di controllo decodificato MD5 di un utente
 *
 * Restituisce il codice di controllo dell'utente passato come parametro,
 * decodificandolo con algoritmo MD5.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $user Nome assoluto del profilo utente da elaborare
 * @return string Codice di controllo decodificato MD5 dell'utente
 */
function getsecid($user) {
	$user = getparam($user,PAR_NULL, SAN_FLAT);
	if(!file_exists("forum/users/".$user.".php"))
		return("");
	$fd = file("forum/users/".$user.".php");
	$pass = str_replace("#","",str_replace("\n","",$fd[1]));
	return (md5($user.$pass));
}


/**
 * Verifica il codice di controllo da FlatNuke
 *
 * Verifica che il codice di controllo immesso dall'utente e salvato nel cookie
 * corrisponda a quello del profilo dell'utente stesso.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $user Nome assoluto del profilo utente da elaborare
 * @return boolean Vero o falso
 */
function versecid($user) {
	$user = getparam($user,PAR_NULL, SAN_FLAT);
	$secid = getparam("secid",PAR_COOKIE, SAN_FLAT);

	if (getsecid($user) == $secid)
		return (TRUE);
	else
		return (FALSE);
}


/**
 * Verifica il codice di controllo dal forum
 *
 * Verifica che il codice di controllo immesso dall'utente e salvato nel cookie
 * corrisponda a quello del profilo dell'utente stesso.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $user Nome assoluto del profilo utente da elaborare
 * @return boolean Vero o falso
 */
function versecid2($user) {
	$user = getparam($user,PAR_NULL, SAN_FLAT);
	$secid = getparam("secid",PAR_COOKIE, SAN_FLAT);

	if (md5($user.getpass($user)) == $secid)
		return (TRUE);
	else
		return (FALSE);
}


/**
 * Mette un file in una stringa
 *
 * Trasforma il file passato come parametro in una stringa di testo.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $filename Nome relativo alla root del file da elaborare
 * @return string Stringa con il contenuto del file
 */
function get_file($filename) {
	$filename = getparam($filename,PAR_NULL, SAN_FLAT);
	$string = "";
	$fd = fopen ("$filename", "r");
	if($fd=="")
		die(_NONPUOI);
	while (!feof ($fd)) {
		$buffer = fgets($fd, 4096);
		$string.=$buffer;
	}
	fclose ($fd);
	return $string;
}


/**
 * Restituisce un elemento XML
 *
 * Restituisce un elemento XML da un file passato come parametro.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $elem Nome dell'elemento XML da cercare
 * @param string $xml Nome del file XML da processare
 * @return string Stringa contenente il valore dell'elemento XML
 */
function get_xml_element($elem,$xml){
	$elem = getparam($elem,PAR_NULL, SAN_FLAT);
	$xml = getparam($xml,PAR_NULL, SAN_FLAT);
	$buff = ereg_replace(".*<".$elem.">","",$xml);
	$buff = ereg_replace("</".$elem.">.*","",$buff);
	return $buff;
}


/**
 * Restituisce l'array di un ramo di elementi XML
 *
 * Restituisce l'array di un ramo di elementi XML da un file passato come parametro.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $elem Nome del ramo di elementi XML da cercare
 * @param string $xml Nome del file XML da processare
 * @return array Array contenente il ramo di elementi XML
 */
function get_xml_array($elem,$xml){
	$elem = getparam($elem,PAR_NULL, SAN_FLAT);
	$xml = getparam($xml,PAR_NULL, SAN_FLAT);
	$buff = explode("</".$elem.">",$xml);
	array_splice ($buff, count($buff)-1);
	return $buff;
}


/**
 * Restituisce il livello di un utente
 *
 * Restituisce il livello dell'utente passato come primo parametro; il secondo
 * parametro serve per gestire il riferimento alla cartella dove sono
 * presenti gli utenti, a seconda che la chiamata parta dall'homepage oppure
 * dal forum:
 *  - 0 per un utente registrato;
 *  - da 1 a 9 per un utente registrato di livello intermedio;
 *  - 10 per un utente amministratore.
 *
 * @author Simone Vellei <simone_vellei@users.sourceforge.net>
 *
 * @param string $admin Nome assoluto del profilo utente da elaborare
 * @param string $from Origine della chiamata ("home"/"forum")
 * @return int Livello dell'utente
 */
function getlevel($admin, $from){
	// controlla implicitamente anche l'esistenza dell'account
	// del nome passato, quindi anche la corrispondenza del cookie
	$admin = getparam($admin,PAR_NULL, SAN_FLAT);
	$from = getparam($from,PAR_NULL, SAN_FLAT);
	if($from=="home"){
		if(!file_exists("forum/users/$admin.php"))
			return(-1);
		$fd = file("forum/users/$admin.php");
		$level = str_replace("#","",str_replace("\n","",$fd[9]));
		return((int)$level);
	}
	if($from=="forum"){
		if(!file_exists("users/$admin.php"))
			return(-1);
		$fd = file("users/$admin.php");
		$level = str_replace("#","",str_replace("\n","",$fd[9]));
		return((int)$level);
	}
}


/**
 * Controllo sulla validit di una stringa alfanumerica
 *
 * Esegue un controllo sulla stringa passata come parametro, verificando che
 * contenga unicamente caratteri oppure numeri, secondo lo standard UNIX.
 *
 * @author Marco Segato <segatom@users.sourceforge.net>
 * @since 2.5.7
 *
 * @param string $string Stringa da verificare
 * @return boolean Vero o Falso
 */
function is_alphanumeric($string) {
	$string = getparam($string,PAR_NULL, SAN_FLAT);
	if(eregi("^[[:alnum:]]+$", $string))
		return (TRUE);
	else return (FALSE);
}


/**
 * Restituisce l'estensione di un file
 *
 * Restituisce l'estensione di un file passato come parametro.
 *
 * @author Marco Segato <segatom@users.sourceforge.net>
 * @since 2.5.7
 *
 * @param string $string Nome del file da verificare
 * @return string $extension Estensione del file
 */
function get_file_extension($filename) {
	$filename = getparam($filename,PAR_NULL, SAN_FLAT);
	ereg('[\.]*[[:alpha:]]+$', $filename, $extension);
	$extension = str_replace(".","",$extension[0]);
	return $extension;
}


/**
 * Restituisce un timestamp formattato in secondi
 *
 * @author Marco Segato <segatom@users.sourceforge.net>
 * @since 2.5.7
 *
 * @return number $mtime Timestamp formattato
 */
function get_microtime() {
	$mtime = microtime();
	$mtime = explode(" ", $mtime);
	$mtime = doubleval($mtime[1]) + doubleval($mtime[0]);

	return $mtime;
}


/**
 * Rende un link W3C compliant
 *
 * @author Aldo Boccacci <zorba_@tin.it>
 * @since 2.5.7
 *
 * @return string $parts URL codificato
 */
function rawurlencodepath($path){
	$parts=array();
	$parts = explode('/', $path);
	for ($i = 0; $i < count($parts); $i++) {
		$parts[$i] = rawurlencode($parts[$i]);
	}
	return implode('/', $parts);
}
?>
