/**
 * \file lib_tree.js
 * \author Mihai OPREA <mihai.oprea@axigen.com>
 * \date 31-07-2006
 *
 * Copyright &copy; 2006 GeCAD Technologies SRL. All Rights Reserved.
 * \brief SCRIPT - Tree
 *
 */

/*******************************************************************************
 * Tree class
 */

function Tree()
{
    this.disableEvents = arguments[1]?arguments[1]:false;
    this.selectFolders = arguments[2]?arguments[2]:false;

    this.selectedFolder = null;
    this.activeFolder = null;

    if (!this.disableEvents) { registerObject(this, "tree"); }

    this.tree = new TreeNode();
    this.started = false;

    if (arguments.length <= 0 || typeof(arguments[0]) != 'object') { return false; }

    this.generate(arguments[0]);
}

Tree.prototype.start = function()
{
    if (this.started) { return false; }

    this.started = true;
}

/** Event handling functions
 */

Tree.prototype.notify = function()
{
    var eventInfo = getEventManager().getEventInfo();

    var eventId = eventInfo[0];
    var callerId = eventInfo[1];

    if (eventId == EVENT_TREE) {
        getManager().openFolder(callerId);
    }
}

/** Generate DOM-like tree structure
 */

Tree.prototype.generate = function(data)
{
    // Setup active folder
    var mailList = getObject("mailList");
    var folderId = mailList?mailList.fid:-1;

    var lastFolder;

    if (this.selectFolders) {
        var rootData = new Object();
        rootData.name = "/";
        rootData.path = "/";
        rootData.fid = "";
        rootData.level = "0";
        rootData.news = "0";
        rootData.count = "0";
        rootData.isSelectable = false;
        
        lastFolder = new TreeNode(rootData, this.disableEvents, this.selectFolders);
        lastFolder.rootNode = this;
        
        this.tree.appendChild(lastFolder);
    }

    lastFolder = new TreeNode(data[0], this.disableEvents, this.selectFolders);
    lastFolder.rootNode = this;
    
    this.tree.appendChild(lastFolder);

    if (-1 != folderId && folderId == lastFolder.data.fid) { this.activeFolder = lastFolder; }

    for (var i = 1; i < data.length; ++i) {
        var folder = new TreeNode(data[i], this.disableEvents, this.selectFolders);
        folder.rootNode = this;

        if (-1 != folderId && folderId == folder.data.fid) { this.activeFolder = folder; }

        if (folder.data.level > lastFolder.data.level) {
            lastFolder.appendChild(folder);

            lastFolder = folder;
        } else if (folder.data.level == lastFolder.data.level) {
            lastFolder.parentNode.appendChild(folder);

            lastFolder = folder;
        } else {
            for (; folder.data.level < lastFolder.data.level && lastFolder.parentNode;
                lastFolder = lastFolder.parentNode);
            
            if (lastFolder.parentNode) {
                lastFolder.parentNode.appendChild(folder);
            } else {
                lastFolder.appendChild(folder);
            }

            lastFolder = folder;
        }
    }
}

/** Draw first level of the tree
 */

Tree.prototype.draw = function(parentContainer)
{
    var body = document.getElementsByTagName("body").item(0);

	if (parentContainer) { body = parentContainer; }

    if (!body) { return false; }

    var container = document.createElement("div");
    container.className = "tree_container";
    
    this.tree.drawChildren(container);
    
    body.appendChild(container);
}

/*******************************************************************************
 * TreeNode class
 */

function TreeNode()
{
    this.data = null;

    this.rootNode = null;
    this.parentNode = null;

    this.childNodes = new Array();

    this.firstChild = null;
    this.lastChild = null;

    this.nextSibling = null;
    this.previousSibling = null;
  
    if (arguments.length <= 0) { return false; }

    this.data = arguments[0];
    
    this.disableEvents = arguments[1]?arguments[1]:false;
    this.selectFolders = arguments[2]?arguments[2]:false;
}

///////////////////////////////////////////////////////
// Append a child to the current node, creating links
///////////////////////////////////////////////////////

TreeNode.prototype.appendChild = function(child)
{
    child.parentNode = this;
   
    this.lastChild = child;
    
    if (this.childNodes.length == 0) {
        this.firstChild = child;
    } else {
        child.previousSibling = this.childNodes[this.childNodes.length - 1];
        this.childNodes[this.childNodes.length - 1].nextSibling = child;
    }

    this.childNodes[this.childNodes.length] = child;
}

/** Draw the curent node (in a given container) before a reference container
 */

TreeNode.prototype.draw = function(container, refContainer)
{
    //////////////////////////////////////
    // Setup current folder HTML element
    //////////////////////////////////////

    var folderElement = document.createElement("div");

    folderElement.className = "tree_folder_container";

    folderElement.onmouseover = function()
    {
        this.className += "_hover";
    }
    
    folderElement.onmouseout = function()
    {
        var idx = this.className.indexOf("_hover");
        if (idx >= 0) { this.className = this.className.substring(0, idx); }
    }

    folderElement.onclick = function()
    {
        if (this.blockEvent) { this.blockEvent = false; return false; }

        // Expand / Collapse folder
        var expandFolder = this.firstChild;
        
        if (!expandFolder) { return false; }

        var idx = expandFolder.className.indexOf("_expand");
        if (idx >= 0) {
            expandFolder.className = expandFolder.className.substring(0, idx);
            expandFolder.className += "_collapse";
        } else {
            idx = expandFolder.className.indexOf("_collapse");
            if (idx >= 0) {
                expandFolder.className = expandFolder.className.substring(0, idx);
                expandFolder.className += "_expand";
            }
        }
        
        if (this.treeNode.childNodes.length) {
            if (this.treeNode.firstChild.htmlContainer) {
                this.treeNode.eraseChildren();
            } else {
                this.treeNode.drawChildren(container, this);
            }
        }

        // Single-select folders section
        if (this.treeNode.selectFolders == "SELECT_SINGLE") {
            if (this.treeNode.rootNode.selectedFolder) {
                var selectedFolder = this.treeNode.rootNode.selectedFolder;
                var idx = selectedFolder.htmlContainer.
                    className.indexOf("_active");
                if (idx >= 0) { selectedFolder.htmlContainer.className = 
                    selectedFolder.htmlContainer.className.substring(0, idx); }
            }
            
            this.treeNode.rootNode.selectedFolder = this.treeNode;

            // Clear hover class
            var idx = this.className.indexOf("_hover");
            if (idx >= 0) { this.className = this.className.substring(0, idx); }

            // Add select class
            this.className += "_active";
        }
    }

    //////////////////////////////////////
    // Setup folder icon / folder content
    //////////////////////////////////////

    var folderData = document.createElement("div");
    
    if (in_array(this.data.name.toLowerCase(), TREE_SPECIAL_FOLDERS)) {
        folderData.className = "tree_folder_" + this.data.name.toLowerCase();
    } else {
        folderData.className = "tree_folder" +
            (!this.data.isSelectable?"_virtual":"");
    }
    
    folderData.onmouseover = function()
    {
        this.className += "_hover";
    }
    
    folderData.onmouseout = function()
    {
        var idx = this.className.indexOf("_hover");
        if (idx >= 0) { this.className = this.className.substring(0, idx); }
    }
    
    if (!this.disableEvents) {
        folderData.onclick = function()
        {
            this.parentNode.blockEvent = true;

            this.treeNode.rootNode.activeFolder = this.treeNode;
    
            getManager().registerEvent(EVENT_TREE, this.parentNode.treeNode);
        }
    }

    var folderExpand = document.createElement("div");
    if (!this.childNodes.length) {
        folderExpand.className = "tree_folder_nochildren";
    } else {
        folderExpand.className = "tree_folder_expand";
    }
    
    var folderText = this.data.isSelectable?
        this.data.name + " (" + this.data.news + "/" + this.data.count +")":
        this.data.name;
    var folderTextNode;

    if (this.data.news != 0) {
        folderTextNode = "<b>" + folderText + "</b>";
    } else {
        folderTextNode = folderText;
    }

    folderData.innerHTML = folderTextNode;
    
    //////////////
    // Link nodes
    //////////////

    folderExpand.style.marginLeft = (this.data.level - 1) * 8 + "px";

    folderElement.appendChild(folderExpand);
    folderElement.appendChild(folderData);

    ////////////////
    // Trash folder
    ////////////////

    if (this.data.path == "Trash") {
        var folderEmpty = document.createElement("a");
        folderEmpty.className = "tree_folder_empty";
        folderEmpty.href = "#";
        folderEmpty.innerHTML = getObject("dictionary").get("empty_trash_link");
        
        folderEmpty.onclick = function()
        {
            var webmailData = getObject("mailList").webmailData;

            if (webmailData.confirmFolderEmpty == "no" ||
                confirm(getObject("dictionary").getAlert("empty_trash_msg")))
            {
                getFrame("frame_tree").document.location.href = "?_h=" + getManager().
                    sessionId + "&page=tree&action=empty_trash";
            }
            
            return false;
        }
        
//        folderElement.appendChild(document.createTextNode("["));
        folderElement.appendChild(folderEmpty);
//        folderElement.appendChild(document.createTextNode("]"));
    }

    if (refContainer) {
        container.insertBefore(folderElement, refContainer);
    } else {
        container.appendChild(folderElement);
    }
    
    this.htmlContainer = folderElement;
    this.container = container;
    folderData.treeNode = this;
    folderElement.treeNode = this;
}

/*******************************************************************************
 * Update current folder news counter
 * (highly dependent upon impl.)
 */

TreeNode.prototype.update = function()
{
    if (!this.htmlContainer) { return false; }

    var oldNews = this.data.news + 1;
    var textNode = this.htmlContainer.childNodes[1].firstChild.firstChild;

    // Public Folder
    if (!textNode) {
        textNode = this.htmlContainer.childNodes[1].firstChild;
    }

    textNode.data = textNode.data.replace(oldNews + "/", this.data.news + "/");
}

/** Draw all child nodes in the given container but before de reference
 * container
 */

TreeNode.prototype.drawChildren = function(container)
{
    var refContainer = this.htmlContainer?this.htmlContainer.nextSibling:null;

    for (var i = 0; i < this.childNodes.length; ++i) {
        var folder = this.childNodes[i];
        folder.draw(container, refContainer);
    }
}

/** Erase all child nodes
 */

TreeNode.prototype.eraseChildren = function()
{
    for (var i = 0; i < this.childNodes.length; ++i) {
        var folder = this.childNodes[i];

        folder.eraseChildren();

        if (!folder.htmlContainer) { continue; }

        folder.container.removeChild(folder.htmlContainer);

        folder.container = null;
        folder.htmlContainer = null;
    }
}

/*******************************************************************************
 * TreeNavigation class
 */

function TreeNavigation()
{
    registerObject(this, "treeNavigation");

    this.started = false;
}

TreeNavigation.prototype.start = function()
{
    if (this.started) { return false; }

    this.started = true;
}

TreeNavigation.prototype.notify = function()
{
    var info = getEventManager().getEventInfo();
    var popup;
    switch(info[1].id) {
        case "folder_new": {
            popup = getManager().openPopup("tree_nav", "&section=new");
            } break;

        case "folder_rename": {
            popup = getManager().openPopup("tree_nav", "&section=rename");
            } break;

        case "folder_copy": {
            popup = getManager().openPopup("tree_nav", "&section=move");
            } break;

        case "folder_delete": {
            popup = getManager().openPopup("tree_nav", "&section=delete");
            } break;
    }
}

TreeNavigation.prototype.draw = function()
{
    var body = document.getElementsByTagName("body").item(0);
    if (!body) { return false; }

    var container = document.createElement("div");
    container.className = "treeNav_container";

    // Create new folder
    var new_folder_container = document.createElement("div");
    new_folder_container.id = "folder_new";
    new_folder_container.className = "treeNav_element";
    getObject("displayManager").addRollOver(new_folder_container, EVENT_TREE_NAV);

    var new_folder = document.createElement("div");
    new_folder.className = "treeNav_new";
    getObject("displayManager").addRollOver(new_folder);

    new_folder_container.appendChild(new_folder);

    // Rename folder
    var ren_folder_container = document.createElement("div");
    ren_folder_container.id = "folder_rename";
    ren_folder_container.className = "treeNav_element";
    getObject("displayManager").addRollOver(ren_folder_container, EVENT_TREE_NAV);

    var ren_folder = document.createElement("div");
    ren_folder.className = "treeNav_ren";
    getObject("displayManager").addRollOver(ren_folder);

    ren_folder_container.appendChild(ren_folder);

    // Copy folder
    var copy_folder_container = document.createElement("div");
    copy_folder_container.id = "folder_copy";
    copy_folder_container.className = "treeNav_element";
    getObject("displayManager").addRollOver(copy_folder_container, EVENT_TREE_NAV);

    var copy_folder = document.createElement("div");
    copy_folder.className = "treeNav_copy";
    getObject("displayManager").addRollOver(copy_folder);

    copy_folder_container.appendChild(copy_folder);

    // Delete folder
    var del_folder_container = document.createElement("div");
    del_folder_container.id = "folder_delete";
    del_folder_container.className = "treeNav_element";
    getObject("displayManager").addRollOver(del_folder_container, EVENT_TREE_NAV);

    var del_folder = document.createElement("div");
    del_folder.className = "treeNav_del";
    getObject("displayManager").addRollOver(del_folder);

    del_folder_container.appendChild(del_folder);

    container.appendChild(new_folder_container);
    container.appendChild(ren_folder_container);
    container.appendChild(copy_folder_container);
    container.appendChild(del_folder_container);

    body.appendChild(container);
}
