/***************************************************************************
 *   Copyright (C) 2005-2008 by Gerald Schmidt                             *
 *   gnschmidt@users.sourceforge.net                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <iostream>
#include <fstream>
#include <string>
#include "wraplibxml.h"
#include "housestyleengine.h"
#include "contexthandler.h"

int main ( int argc, char *argv[] )
{
	if (argc < 8)
	{
		std::cout << "Usage: housestyle <file> <style directory> <style> <xpath> <format> <raw output> <formatted output> [<display name>]"; // use std::cout so PHP's system call picks it up
		return EXIT_FAILURE;
	}
	std::string file, style, styleDirectory, xpath, format, rawOutput, formattedOutput, displayName;
	file = *(++argv);
	styleDirectory = *(++argv);
	style = *(++argv);
	xpath = *(++argv);
	format = *(++argv);
	rawOutput = *(++argv);
	formattedOutput = *(++argv);
	if (argc == 9)
		displayName = *(++argv);

	std::vector<std::pair<std::string, unsigned> > nodeVector;
	std::auto_ptr<WrapLibxml> wrapLibxml ( new WrapLibxml () );
	if (!wrapLibxml.get()->getPathVector ( xpath, file, nodeVector ) )
	{
		std::cout << "Cannot parse file: " << wrapLibxml.get()->getLastError();
		return EXIT_FAILURE;
	}

	std::auto_ptr<HouseStyleEngine> houseStyleEngine ( new HouseStyleEngine (
		styleDirectory,
		style,
		"/",
		5) ); // should context range be user-definable?
	if (!houseStyleEngine.get()->createReport ( nodeVector ) )
	{
		std::cout << "Cannot apply style file " << style.c_str() << ".";
		return EXIT_FAILURE;
	}
	
	std::vector<ContextMatch> matchVector = houseStyleEngine.get()->getMatchVector();

	if (matchVector.empty())
	{
		std::cout << "Success: no problems found.";
		return EXIT_FAILURE;
	}

	std::ofstream outputStream ( rawOutput.c_str() );
	if (!outputStream)
	{
		std::cout << "Cannot create output file.";
		return EXIT_FAILURE;
	}

	if (!displayName.empty())
		file = displayName;
	outputStream << "<report>" << "<header>";
	outputStream << "<file>" << file.c_str() << "</file>";
	outputStream << "<style>" << style.c_str() << "</style>";
	outputStream << "<errors>" << matchVector.size() << "</errors>";
	outputStream << "</header>";

	std::vector<ContextMatch>::iterator it;
	ContextHandler handler;
	size_t count = 0;
	for (it = matchVector.begin(); it != matchVector.end(); it++)
	{	
		handler.normaliseMatch ( *it );
		outputStream << "<record>";
		outputStream << "<count>" << ++count << "</count>";
		outputStream << "<precedingtext>" << it->prelog.c_str() << "</precedingtext>";
		outputStream << "<match>" << it->match.c_str() << "</match>";
		outputStream << "<followingtext>" << it->postlog.c_str() << "</followingtext>";
		outputStream << "<replacement>" << it->replace.c_str() << "</replacement>";
		outputStream << "<comment>" << it->report.c_str() << "</comment>";
		outputStream << "<tentative>";
		if (it->tentative)
			outputStream << "true";
		else
			outputStream << "false";
		outputStream << "</tentative></record>";
	}
	outputStream << "</report>";
	outputStream.close();

	if (!wrapLibxml.get()->xslt( format, rawOutput ) )
	{
		std::cout << "Cannot transform output: " << wrapLibxml.get()->getLastError().c_str();
		return EXIT_FAILURE;
	}
	std::ofstream formattedOutputStream ( formattedOutput.c_str() );
	if (!formattedOutputStream)
	{
		std::cout << "Cannot save file containing formatted output.";
		return EXIT_FAILURE;
	}
	formattedOutputStream << wrapLibxml.get()->getOutput() << std::endl;
	
	printf ( "The check has identified %u potential problem%s.", matchVector.size(), (matchVector.size() == 1) ? "" : "s");

	return 0;
}
