/*
 * Copyright 2005-2007 Gerald Schmidt.
 *
 * This file is part of Xml Copy Editor.
 *
 * Xml Copy Editor is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * Xml Copy Editor is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Xml Copy Editor; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdexcept>
#include "housestyleengine.h"
#include "readfile.h"

HouseStyleEngine::HouseStyleEngine (
    const std::string& ruleDirectoryParameter,
    const std::string& ruleFileParameter,
    const std::string& pathSeparatorParameter,
    int contextRangeParameter ) :
		ruleDirectory ( ruleDirectoryParameter ),
		ruleFile ( ruleFileParameter ),
		pathSeparator ( pathSeparatorParameter ),
		contextRange ( contextRangeParameter ),
		ruleVector ( new std::vector<boost::shared_ptr<Rule> > )
{}

HouseStyleEngine::~HouseStyleEngine()
{}

void HouseStyleEngine::collectRules ( string& fileName,
                                boost::shared_ptr<std::vector<boost::shared_ptr<Rule> > > ruleVector,
                                std::set<string>& excludeSet,
                                int *ruleCount )
{
	std::string filePath, buffer;
	filePath = ruleDirectory + pathSeparator + fileName;
	if ( !ReadFile::run ( filePath, buffer ) )
		return;

	std::auto_ptr<XmlRuleReader> xrr ( new XmlRuleReader (
	                                       ruleVector ) );
	if ( !xrr->parse ( buffer ) )
	{
		std::string report = xrr->getIncorrectPatternReport();
		if ( report != "" )
			throw runtime_error ( report.c_str() );
		else
			throw runtime_error ( xrr->getLastError().c_str() );
	}

	// add current file to exclude set
	excludeSet.insert ( fileName );

	// fetch exclude vector
	std::vector<std::string> localExcludeVector;
	std::vector<std::string>::iterator excludeIterator;
	xrr->getExcludeVector ( localExcludeVector );
	for ( excludeIterator = localExcludeVector.begin();
	        excludeIterator != localExcludeVector.end();
	        excludeIterator++ )
		excludeSet.insert ( *excludeIterator );

	* ( ruleCount ) += xrr->getRuleCount();

	// fetch include vector
	std::vector<std::string> includeVector;
	xrr->getIncludeVector ( includeVector );
	std::vector<std::string>::iterator includeIterator;
	for ( includeIterator = includeVector.begin();
	        includeIterator != includeVector.end();
	        includeIterator++ )
	{
		if ( !excludeSet.count ( *includeIterator ) )
			collectRules ( *includeIterator, ruleVector, excludeSet, ruleCount );
	}
}

bool HouseStyleEngine::createReport( const std::vector<std::pair<std::string, unsigned> >& nodeVector )
{
	if ( !updateRules() )
	{
		error = "no rules found";
		return false;
	}
	int ruleVectorsize, nodeVectorSize;
	std::vector<ContextMatch> contextVector;
	std::vector<ContextMatch>::iterator matchIterator;
	ruleVectorsize = ruleVector->size();

	nodeVectorSize = nodeVector.size();

	std::string nodeBuffer;
	unsigned elementCount;
	for ( int j = 0; j < nodeVectorSize; ++j )
	{
		nodeBuffer = nodeVector.at ( j ).first;
		elementCount = nodeVector.at ( j ).second;

		if ( !nodeBuffer.size() )
			continue;

		// otherwise, proceed with style check
		for ( int i = 0; i < ruleVectorsize; i++ )
		{
			boost::shared_ptr<Rule> rule ( ruleVector->at ( i ) );
			if ( rule->matchPatternGlobal (
				nodeBuffer,
				contextVector,
				elementCount,
				contextRange ) )
			{
				std::string report = rule->getReport();

				for ( matchIterator = contextVector.begin();
					matchIterator != contextVector.end();
					matchIterator++ )
				{
					if ( rule->getAdjustCaseAttribute() )
						CaseHandler::adjustCase (
						matchIterator->replace,
						matchIterator->match );

					// tentative?
					matchIterator->tentative =
					( rule->getTentativeAttribute() ) ? true : false;
					matchIterator->report = report;

					matchVector.push_back ( *matchIterator );
				}
				contextVector.clear();
			}
		}
	}
	return true;
}

std::string HouseStyleEngine::getLastError()
{
	return error;
}

std::vector<ContextMatch> HouseStyleEngine::getMatchVector()
{
	return matchVector;
}

int HouseStyleEngine::updateRules()
{
	ruleVector->clear();

	int ruleCount = 0;
	set<string> excludeSet;
	collectRules ( ruleFile, ruleVector, excludeSet, &ruleCount );
	return ruleCount;
}
